/*
 * Copyright 2010-2023 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.analysis.api.fir.types

import ksp.org.jetbrains.kotlin.analysis.api.KaExperimentalApi
import ksp.org.jetbrains.kotlin.analysis.api.annotations.KaAnnotationList
import ksp.org.jetbrains.kotlin.analysis.api.fir.KaSymbolByFirBuilder
import ksp.org.jetbrains.kotlin.analysis.api.fir.annotations.KaFirAnnotationListForType
import ksp.org.jetbrains.kotlin.analysis.api.fir.utils.createTypePointer
import ksp.org.jetbrains.kotlin.analysis.api.lifetime.KaLifetimeToken
import ksp.org.jetbrains.kotlin.analysis.api.lifetime.withValidityAssertion
import ksp.org.jetbrains.kotlin.analysis.api.types.*
import ksp.org.jetbrains.kotlin.fir.types.ConeIntersectionType
import ksp.org.jetbrains.kotlin.fir.types.renderForDebugging

internal class KaFirIntersectionType(
    override val coneType: ConeIntersectionType,
    private val builder: KaSymbolByFirBuilder,
) : KaIntersectionType(), KaFirType {
    override val token: KaLifetimeToken get() = builder.token

    override val conjuncts: List<KaType>
        get() = withValidityAssertion {
            coneType.intersectedTypes.map { conjunct -> builder.typeBuilder.buildKtType(conjunct) }
        }

    override val annotations: KaAnnotationList
        get() = withValidityAssertion {
            KaFirAnnotationListForType.create(coneType, builder)
        }

    @Deprecated(
        "Use `isMarkedNullable`, `isNullable` or `hasFlexibleNullability` instead. See KDocs for the migration guide",
        replaceWith = ReplaceWith("this.isMarkedNullable")
    )
    @Suppress("Deprecation")
    override val nullability: KaTypeNullability get() = withValidityAssertion { KaTypeNullability.NON_NULLABLE }

    override val abbreviation: KaUsualClassType?
        get() = withValidityAssertion { null }

    override fun equals(other: Any?) = typeEquals(other)
    override fun hashCode() = typeHashcode()
    override fun toString() = coneType.renderForDebugging()

    @KaExperimentalApi
    override fun createPointer(): KaTypePointer<KaIntersectionType> = withValidityAssertion {
        return createTypePointer(coneType, builder, ::KaFirIntersectionType)
    }
}