/*
 * Copyright 2010-2025 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.symbolProviders

import ksp.com.intellij.psi.PsiClass
import ksp.com.intellij.psi.PsiElement
import ksp.com.intellij.psi.search.GlobalSearchScope
import ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.projectStructure.moduleData
import ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.sessions.LLFirSession
import ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.symbolProviders.caches.LLPsiAwareClassLikeSymbolCache
import ksp.org.jetbrains.kotlin.fir.FirSession
import ksp.org.jetbrains.kotlin.fir.caches.firCachesFactory
import ksp.org.jetbrains.kotlin.fir.java.FirJavaFacade
import ksp.org.jetbrains.kotlin.fir.java.FirJavaFacadeForSource
import ksp.org.jetbrains.kotlin.fir.java.JavaSymbolProvider
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirRegularClassSymbol
import ksp.org.jetbrains.kotlin.load.java.createJavaClassFinder
import ksp.org.jetbrains.kotlin.load.java.structure.impl.JavaClassImpl
import ksp.org.jetbrains.kotlin.name.ClassId

internal class LLFirJavaSymbolProvider private constructor(
    session: LLFirSession,
    javaFacade: FirJavaFacade,
    val searchScope: GlobalSearchScope
) : JavaSymbolProvider(session, javaFacade), LLPsiAwareSymbolProvider {
    constructor(session: LLFirSession, searchScope: GlobalSearchScope) : this(
        session,
        FirJavaFacadeForSource(
            session,
            session.moduleData,
            session.project.createJavaClassFinder(searchScope)
        ),
        searchScope
    )

    private val psiAwareCache = LLPsiAwareClassLikeSymbolCache<PsiClass, FirRegularClassSymbol?, ClassCacheContext?>(
        classCache,
        session.firCachesFactory.createCache { psiClass, classCacheContext ->
            javaFacade.createPsiClassSymbol(psiClass, classCacheContext?.foundJavaClass, classCacheContext?.parentClassSymbol)
        }
    )

    @LLModuleSpecificSymbolProviderAccess
    override fun getClassLikeSymbolByPsi(classId: ClassId, declaration: PsiElement): FirRegularClassSymbol? =
        psiAwareCache.getSymbolByPsi<PsiClass>(classId, declaration) { psiClass ->
            val parentClass = getParentPsiClassSymbol(psiClass)
            ClassCacheContext(parentClass, JavaClassImpl(psiClass))
        }
}

internal val FirSession.nullableJavaSymbolProvider: JavaSymbolProvider? by FirSession.nullableSessionComponentAccessor()
