/*
 * Copyright 2010-2024 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.backend.common.serialization

import ksp.org.jetbrains.kotlin.builtins.FunctionInterfacePackageFragment
import ksp.org.jetbrains.kotlin.ir.IrDiagnosticReporter
import ksp.org.jetbrains.kotlin.ir.declarations.IrFile
import ksp.org.jetbrains.kotlin.ir.declarations.IrModuleFragment
import ksp.org.jetbrains.kotlin.ir.declarations.IrSimpleFunction
import ksp.org.jetbrains.kotlin.ir.declarations.packageFragmentDescriptor
import ksp.org.jetbrains.kotlin.ir.util.preparedInlineFunctionCopies
import ksp.org.jetbrains.kotlin.library.SerializedIrFile
import ksp.org.jetbrains.kotlin.library.SerializedIrModule

abstract class IrModuleSerializer<Serializer : IrFileSerializer>(
    protected val settings: IrSerializationSettings,
    protected val diagnosticReporter: IrDiagnosticReporter,
) {
    abstract fun createFileSerializer(): Serializer

    /**
     * Allows to skip [file] during serialization.
     *
     * For example, some files should be generated anew instead of deserialization.
     */
    protected open fun backendSpecificFileFilter(file: IrFile): Boolean =
        true

    protected abstract val globalDeclarationTable: GlobalDeclarationTable

    private fun serializeIrFile(file: IrFile): SerializedIrFile {
        val fileSerializer = createFileSerializer()
        return fileSerializer.serializeIrFile(file)
    }

    private fun serializePreparedInlinableFunctions(preparedInlineFunctionCopies: List<IrSimpleFunction>): SerializedIrFile {
        val fileSerializer = createFileSerializer()
        return fileSerializer.serializeIrFileWithPreparedInlineFunctions(preparedInlineFunctionCopies)
    }

    fun serializedIrModule(module: IrModuleFragment): SerializedIrModule {
        val serializedFiles = module.files
            .filter { it.packageFragmentDescriptor !is FunctionInterfacePackageFragment }
            .filter(this::backendSpecificFileFilter)
            .map(this::serializeIrFile)
        if (settings.shouldCheckSignaturesOnUniqueness) {
            globalDeclarationTable.clashDetector.reportErrorsTo(diagnosticReporter)
        }

        val inlinableFunctionsFile = module.preparedInlineFunctionCopies?.let {
            serializePreparedInlinableFunctions(it)
        }
        return SerializedIrModule(serializedFiles, inlinableFunctionsFile)
    }
}