/*
 * Copyright 2010-2025 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.light.classes.symbol.fields

import ksp.com.intellij.psi.PsiExpression
import ksp.com.intellij.psi.PsiModifier
import ksp.com.intellij.psi.PsiModifierList
import ksp.com.intellij.psi.PsiType
import ksp.kotlinx.collections.immutable.mutate
import ksp.org.jetbrains.annotations.NotNull
import ksp.org.jetbrains.kotlin.analysis.api.KaSession
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaNamedClassSymbol
import ksp.org.jetbrains.kotlin.analysis.api.symbols.pointers.KaSymbolPointer
import ksp.org.jetbrains.kotlin.analysis.api.symbols.sourcePsiSafe
import ksp.org.jetbrains.kotlin.asJava.builder.LightMemberOrigin
import ksp.org.jetbrains.kotlin.asJava.classes.lazyPub
import ksp.org.jetbrains.kotlin.light.classes.symbol.*
import ksp.org.jetbrains.kotlin.light.classes.symbol.annotations.ComputeAllAtOnceAnnotationsBox
import ksp.org.jetbrains.kotlin.light.classes.symbol.annotations.SymbolLightSimpleAnnotation
import ksp.org.jetbrains.kotlin.light.classes.symbol.annotations.hasDeprecatedAnnotation
import ksp.org.jetbrains.kotlin.light.classes.symbol.classes.SymbolLightClassForClassLike
import ksp.org.jetbrains.kotlin.light.classes.symbol.modifierLists.GranularModifiersBox
import ksp.org.jetbrains.kotlin.light.classes.symbol.modifierLists.InitializedModifiersBox
import ksp.org.jetbrains.kotlin.light.classes.symbol.modifierLists.SymbolLightMemberModifierList
import ksp.org.jetbrains.kotlin.psi.KtObjectDeclaration

internal class SymbolLightFieldForObject private constructor(
    containingClass: SymbolLightClassForClassLike<*>,
    private val name: String,
    lightMemberOrigin: LightMemberOrigin?,
    private val objectSymbolPointer: KaSymbolPointer<KaNamedClassSymbol>,
    override val kotlinOrigin: KtObjectDeclaration?,
    private val isCompanion: Boolean,
) : SymbolLightField(containingClass, lightMemberOrigin) {
    internal constructor(
        objectSymbol: KaNamedClassSymbol,
        name: String,
        lightMemberOrigin: LightMemberOrigin?,
        containingClass: SymbolLightClassForClassLike<*>,
        isCompanion: Boolean,
    ) : this(
        containingClass = containingClass,
        name = name,
        lightMemberOrigin = lightMemberOrigin,
        kotlinOrigin = objectSymbol.sourcePsiSafe(),
        objectSymbolPointer = objectSymbol.createPointer(),
        isCompanion = isCompanion,
    )

    private inline fun <T> withObjectDeclarationSymbol(crossinline action: KaSession.(KaNamedClassSymbol) -> T): T =
        objectSymbolPointer.withSymbol(ktModule, action)

    override fun getName(): String = name

    override fun getModifierList(): PsiModifierList = cachedValue {
        SymbolLightMemberModifierList(
            containingDeclaration = this,
            modifiersBox = if (isCompanion) {
                GranularModifiersBox(
                    initialValue = GranularModifiersBox.MODALITY_MODIFIERS_MAP.mutate {
                        it[PsiModifier.FINAL] = true
                        it[PsiModifier.STATIC] = true
                    },
                    computer = ::computeCompanionModifiers,
                )
            } else {
                InitializedModifiersBox(PsiModifier.PUBLIC, PsiModifier.STATIC, PsiModifier.FINAL)
            },
            annotationsBox = ComputeAllAtOnceAnnotationsBox { modifierList ->
                listOf(SymbolLightSimpleAnnotation(NotNull::class.java.name, modifierList))
            },
        )
    }

    private fun computeCompanionModifiers(modifier: String): Map<String, Boolean>? {
        if (modifier !in GranularModifiersBox.VISIBILITY_MODIFIERS) return null
        return GranularModifiersBox.computeVisibilityForClass(ktModule, objectSymbolPointer, isTopLevel = false)
    }

    private val _isDeprecated: Boolean by lazyPub {
        withObjectDeclarationSymbol { objectSymbol ->
            objectSymbol.hasDeprecatedAnnotation()
        }
    }

    override fun isDeprecated(): Boolean = _isDeprecated

    private val _type: PsiType by lazyPub {
        withObjectDeclarationSymbol { objectSymbol ->
            objectSymbol.defaultType
                .asPsiType(
                    this@SymbolLightFieldForObject,
                    allowErrorTypes = true,
                    allowNonJvmPlatforms = true,
                )
        } ?: nonExistentType()
    }

    override fun getType(): PsiType = _type

    override fun getInitializer(): PsiExpression? = null //TODO

    override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (other !is SymbolLightFieldForObject || other.ktModule != ktModule) return false
        if (kotlinOrigin != null || other.kotlinOrigin != null) {
            return other.kotlinOrigin == kotlinOrigin
        }

        return other.containingClass == containingClass &&
                compareSymbolPointers(other.objectSymbolPointer, objectSymbolPointer)
    }

    override fun hashCode(): Int = kotlinOrigin.hashCode()

    override fun isValid(): Boolean = kotlinOrigin?.isValid ?: objectSymbolPointer.isValid(ktModule)
}
