/*
 * Copyright 2010-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package ksp.org.jetbrains.kotlin.load.java.structure.impl;

import ksp.com.intellij.psi.PsiEnumConstant;
import ksp.com.intellij.psi.PsiField;
import ksp.org.jetbrains.annotations.NotNull;
import ksp.org.jetbrains.annotations.Nullable;
import ksp.org.jetbrains.kotlin.load.java.structure.JavaField;
import ksp.org.jetbrains.kotlin.load.java.structure.JavaType;
import ksp.org.jetbrains.kotlin.load.java.structure.impl.source.JavaElementPsiSource;
import ksp.org.jetbrains.kotlin.load.java.structure.impl.source.JavaElementSourceFactory;

import static org.jetbrains.kotlin.util.ConstUtils.isCompileTimeConstant;

public class JavaFieldImpl extends JavaMemberImpl<PsiField> implements JavaField {
    public JavaFieldImpl(@NotNull JavaElementPsiSource<PsiField> psiFieldSource) {
        super(psiFieldSource);
    }

    @SuppressWarnings("unused") // used in KSP
    public JavaFieldImpl(PsiField psiField) {
        this(JavaElementSourceFactory.getInstance(psiField.getProject()).createPsiSource(psiField));
    }

    @Override
    public boolean isEnumEntry() {
        return getPsi() instanceof PsiEnumConstant;
    }

    @Override
    @NotNull
    public JavaType getType() {
        return JavaTypeImpl.create(psiElementSource.getPsi().getType(), createVariableReturnTypeSource(psiElementSource));
    }

    @Nullable
    @Override
    public Object getInitializerValue() {
        return getPsi().computeConstantValue();
    }

    @Override
    public boolean getHasConstantNotNullInitializer() {
        // PsiUtil.isCompileTimeConstant returns false for null-initialized fields,
        // see IsConstantExpressionVisitor.visitLiteralExpression()
        return isCompileTimeConstant(getPsi());
    }
}
