/*
 * Copyright 2010-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package ksp.org.jetbrains.kotlin.types.expressions

import ksp.com.intellij.psi.PsiElement
import ksp.com.intellij.psi.util.PsiTreeUtil
import ksp.org.jetbrains.kotlin.cfg.ControlFlowInformationProvider
import ksp.org.jetbrains.kotlin.config.LanguageVersionSettings
import ksp.org.jetbrains.kotlin.container.get
import ksp.org.jetbrains.kotlin.context.GlobalContext
import ksp.org.jetbrains.kotlin.context.withModule
import ksp.org.jetbrains.kotlin.context.withProject
import ksp.org.jetbrains.kotlin.descriptors.ClassDescriptor
import ksp.org.jetbrains.kotlin.descriptors.DeclarationDescriptor
import ksp.org.jetbrains.kotlin.descriptors.ModuleDescriptor
import ksp.org.jetbrains.kotlin.descriptors.SupertypeLoopChecker
import ksp.org.jetbrains.kotlin.frontend.di.createContainerForLazyLocalClassifierAnalyzer
import ksp.org.jetbrains.kotlin.incremental.components.LookupLocation
import ksp.org.jetbrains.kotlin.incremental.components.LookupTracker
import ksp.org.jetbrains.kotlin.lexer.KtTokens
import ksp.org.jetbrains.kotlin.name.FqName
import ksp.org.jetbrains.kotlin.platform.TargetPlatform
import ksp.org.jetbrains.kotlin.psi.KtClassOrObject
import ksp.org.jetbrains.kotlin.psi.KtFile
import ksp.org.jetbrains.kotlin.psi.debugText.getDebugText
import ksp.org.jetbrains.kotlin.resolve.*
import ksp.org.jetbrains.kotlin.resolve.calls.smartcasts.DataFlowInfo
import ksp.org.jetbrains.kotlin.resolve.extensions.SyntheticResolveExtension
import ksp.org.jetbrains.kotlin.resolve.lazy.*
import ksp.org.jetbrains.kotlin.resolve.lazy.data.KtClassInfoUtil
import ksp.org.jetbrains.kotlin.resolve.lazy.data.KtClassLikeInfo
import ksp.org.jetbrains.kotlin.resolve.lazy.declarations.ClassMemberDeclarationProvider
import ksp.org.jetbrains.kotlin.resolve.lazy.declarations.DeclarationProviderFactory
import ksp.org.jetbrains.kotlin.resolve.lazy.declarations.PackageMemberDeclarationProvider
import ksp.org.jetbrains.kotlin.resolve.lazy.declarations.PsiBasedClassMemberDeclarationProvider
import ksp.org.jetbrains.kotlin.resolve.lazy.descriptors.LazyClassDescriptor
import ksp.org.jetbrains.kotlin.resolve.sam.SamConversionResolver
import ksp.org.jetbrains.kotlin.resolve.scopes.LexicalScope
import ksp.org.jetbrains.kotlin.resolve.scopes.LexicalWritableScope
import ksp.org.jetbrains.kotlin.storage.StorageManager
import ksp.org.jetbrains.kotlin.types.WrappedTypeFactory
import ksp.org.jetbrains.kotlin.types.checker.NewKotlinTypeChecker

class LocalClassifierAnalyzer(
    private val globalContext: GlobalContext,
    private val storageManager: StorageManager,
    private val descriptorResolver: DescriptorResolver,
    private val functionDescriptorResolver: FunctionDescriptorResolver,
    private val typeResolver: TypeResolver,
    private val annotationResolver: AnnotationResolver,
    private val platform: TargetPlatform,
    private val analyzerServices: PlatformDependentAnalyzerServices,
    private val lookupTracker: LookupTracker,
    private val supertypeLoopChecker: SupertypeLoopChecker,
    private val languageVersionSettings: LanguageVersionSettings,
    private val delegationFilter: DelegationFilter,
    private val wrappedTypeFactory: WrappedTypeFactory,
    private val kotlinTypeChecker: NewKotlinTypeChecker,
    private val samConversionResolver: SamConversionResolver,
    private val additionalClassPartsProvider: AdditionalClassPartsProvider,
    private val sealedClassInheritorsProvider: SealedClassInheritorsProvider,
    private val controlFlowInformationProviderFactory: ControlFlowInformationProvider.Factory,
    private val absentDescriptorHandler: AbsentDescriptorHandler
) {
    fun processClassOrObject(
        scope: LexicalWritableScope?,
        context: ExpressionTypingContext,
        containingDeclaration: DeclarationDescriptor,
        classOrObject: KtClassOrObject
    ) {
        val module = DescriptorUtils.getContainingModule(containingDeclaration)
        val project = classOrObject.project
        val moduleContext = globalContext.withProject(project).withModule(module)
        val container = createContainerForLazyLocalClassifierAnalyzer(
            moduleContext,
            context.trace,
            platform,
            lookupTracker,
            languageVersionSettings,
            context.statementFilter,
            LocalClassDescriptorHolder(
                scope,
                classOrObject,
                containingDeclaration,
                storageManager,
                context,
                module,
                descriptorResolver,
                functionDescriptorResolver,
                typeResolver,
                annotationResolver,
                supertypeLoopChecker,
                languageVersionSettings,
                SyntheticResolveExtension.getInstance(project),
                delegationFilter,
                wrappedTypeFactory,
                kotlinTypeChecker,
                samConversionResolver,
                additionalClassPartsProvider,
                sealedClassInheritorsProvider
            ),
            analyzerServices,
            controlFlowInformationProviderFactory,
            absentDescriptorHandler
        )

        @Suppress("DEPRECATION")
        container.get<LazyTopDownAnalyzer>().analyzeDeclarations(
            TopDownAnalysisMode.LocalDeclarations,
            listOf(classOrObject),
            context.dataFlowInfo,
            localContext = context
        )
    }
}

class LocalClassDescriptorHolder(
    val writableScope: LexicalWritableScope?,
    val myClass: KtClassOrObject,
    val containingDeclaration: DeclarationDescriptor,
    val storageManager: StorageManager,
    val expressionTypingContext: ExpressionTypingContext,
    val moduleDescriptor: ModuleDescriptor,
    val descriptorResolver: DescriptorResolver,
    val functionDescriptorResolver: FunctionDescriptorResolver,
    val typeResolver: TypeResolver,
    val annotationResolver: AnnotationResolver,
    val supertypeLoopChecker: SupertypeLoopChecker,
    val languageVersionSettings: LanguageVersionSettings,
    val syntheticResolveExtension: SyntheticResolveExtension,
    val delegationFilter: DelegationFilter,
    val wrappedTypeFactory: WrappedTypeFactory,
    val kotlinTypeChecker: NewKotlinTypeChecker,
    val samConversionResolver: SamConversionResolver,
    val additionalClassPartsProvider: AdditionalClassPartsProvider,
    val sealedClassInheritorsProvider: SealedClassInheritorsProvider
) {
    // We do not need to synchronize here, because this code is used strictly from one thread
    private var classDescriptor: ClassDescriptor? = null

    fun isMyClass(element: PsiElement): Boolean = element == myClass
    fun insideMyClass(element: PsiElement): Boolean = PsiTreeUtil.isAncestor(myClass, element, false)

    fun getClassDescriptor(classOrObject: KtClassOrObject, declarationScopeProvider: DeclarationScopeProvider): ClassDescriptor {
        assert(isMyClass(classOrObject)) { "Called on a wrong class: ${classOrObject.getDebugText()}" }
        if (classDescriptor == null) {
            classDescriptor = LazyClassDescriptor(
                object : LazyClassContext {
                    override val declarationScopeProvider = declarationScopeProvider
                    override val inferenceSession = expressionTypingContext.inferenceSession
                    override val storageManager = this@LocalClassDescriptorHolder.storageManager
                    override val trace = expressionTypingContext.trace
                    override val moduleDescriptor = this@LocalClassDescriptorHolder.moduleDescriptor
                    override val descriptorResolver = this@LocalClassDescriptorHolder.descriptorResolver
                    override val functionDescriptorResolver = this@LocalClassDescriptorHolder.functionDescriptorResolver
                    override val typeResolver = this@LocalClassDescriptorHolder.typeResolver
                    override val declarationProviderFactory = object : DeclarationProviderFactory {
                        override fun getClassMemberDeclarationProvider(classLikeInfo: KtClassLikeInfo): ClassMemberDeclarationProvider {
                            return PsiBasedClassMemberDeclarationProvider(storageManager, classLikeInfo)
                        }

                        override fun getPackageMemberDeclarationProvider(packageFqName: FqName): PackageMemberDeclarationProvider? {
                            throw UnsupportedOperationException("Should not be called for top-level declarations")
                        }

                        override fun diagnoseMissingPackageFragment(fqName: FqName, file: KtFile?) {
                            throw UnsupportedOperationException()
                        }
                    }
                    override val annotationResolver = this@LocalClassDescriptorHolder.annotationResolver
                    override val lookupTracker: LookupTracker = LookupTracker.DO_NOTHING
                    override val supertypeLoopChecker = this@LocalClassDescriptorHolder.supertypeLoopChecker
                    override val languageVersionSettings = this@LocalClassDescriptorHolder.languageVersionSettings
                    override val syntheticResolveExtension = this@LocalClassDescriptorHolder.syntheticResolveExtension
                    override val delegationFilter: DelegationFilter = this@LocalClassDescriptorHolder.delegationFilter
                    override val wrappedTypeFactory: WrappedTypeFactory = this@LocalClassDescriptorHolder.wrappedTypeFactory
                    override val kotlinTypeCheckerOfOwnerModule: NewKotlinTypeChecker = this@LocalClassDescriptorHolder.kotlinTypeChecker
                    override val samConversionResolver: SamConversionResolver = this@LocalClassDescriptorHolder.samConversionResolver
                    override val additionalClassPartsProvider: AdditionalClassPartsProvider =
                        this@LocalClassDescriptorHolder.additionalClassPartsProvider
                    override val sealedClassInheritorsProvider: SealedClassInheritorsProvider =
                        this@LocalClassDescriptorHolder.sealedClassInheritorsProvider
                },
                containingDeclaration,
                classOrObject.nameAsSafeName,
                KtClassInfoUtil.createClassOrObjectInfo(classOrObject),
                classOrObject.hasModifier(KtTokens.EXTERNAL_KEYWORD)
            )
            writableScope?.addClassifierDescriptor(classDescriptor!!)
        }

        return classDescriptor!!
    }

    fun getResolutionScopeForClass(classOrObject: KtClassOrObject): LexicalScope {
        assert(isMyClass(classOrObject)) { "Called on a wrong class: ${classOrObject.getDebugText()}" }
        return expressionTypingContext.scope
    }
}

class LocalLazyDeclarationResolver(
    globalContext: GlobalContext,
    trace: BindingTrace,
    private val localClassDescriptorManager: LocalClassDescriptorHolder,
    topLevelDescriptorProvider: TopLevelDescriptorProvider,
    absentDescriptorHandler: AbsentDescriptorHandler
) : LazyDeclarationResolver(globalContext, trace, topLevelDescriptorProvider, absentDescriptorHandler) {

    override fun getClassDescriptor(classOrObject: KtClassOrObject, location: LookupLocation): ClassDescriptor {
        if (localClassDescriptorManager.isMyClass(classOrObject)) {
            return localClassDescriptorManager.getClassDescriptor(classOrObject, scopeProvider)
        }
        return super.getClassDescriptor(classOrObject, location)
    }

    override fun getClassDescriptorIfAny(classOrObject: KtClassOrObject, location: LookupLocation): ClassDescriptor? {
        if (localClassDescriptorManager.isMyClass(classOrObject)) {
            return localClassDescriptorManager.getClassDescriptor(classOrObject, scopeProvider)
        }
        return super.getClassDescriptorIfAny(classOrObject, location)
    }
}


class DeclarationScopeProviderForLocalClassifierAnalyzer(
    lazyDeclarationResolver: LazyDeclarationResolver,
    fileScopeProvider: FileScopeProvider,
    private val localClassDescriptorManager: LocalClassDescriptorHolder
) : DeclarationScopeProviderImpl(lazyDeclarationResolver, fileScopeProvider) {
    override fun getResolutionScopeForDeclaration(elementOfDeclaration: PsiElement): LexicalScope {
        if (localClassDescriptorManager.isMyClass(elementOfDeclaration)) {
            return localClassDescriptorManager.getResolutionScopeForClass(elementOfDeclaration as KtClassOrObject)
        }
        return super.getResolutionScopeForDeclaration(elementOfDeclaration)
    }

    override fun getOuterDataFlowInfoForDeclaration(elementOfDeclaration: PsiElement): DataFlowInfo {
        // nested (non-inner) classes and companion objects are forbidden in local classes, so it's enough to be simply inside the class
        if (localClassDescriptorManager.insideMyClass(elementOfDeclaration)) {
            return localClassDescriptorManager.expressionTypingContext.dataFlowInfo
        }
        return super.getOuterDataFlowInfoForDeclaration(elementOfDeclaration)
    }
}
