// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package ksp.com.intellij.psi.util;

import ksp.com.intellij.openapi.util.text.StringUtil;
import ksp.com.intellij.psi.PsiClass;
import ksp.com.intellij.psi.PsiElement;
import ksp.com.intellij.psi.PsiImplicitClass;
import ksp.com.intellij.psi.PsiJavaFile;
import ksp.org.jetbrains.annotations.NotNull;
import ksp.org.jetbrains.annotations.Nullable;

public final class JavaImplicitClassUtil {
  public static boolean isFileWithImplicitClass(@NotNull PsiElement file) {
    return getImplicitClassFor(file) != null;
  }

  /**
   * Retrieves the implicitly declared class PSI element from the given PsiFile.
   *
   * @param file the PsiFile from which to retrieve the implicitly declared class
   * @return the implicitly declared class if found, null otherwise
   */
  @Nullable
  public static PsiImplicitClass getImplicitClassFor(@NotNull PsiElement file) {
    if (file instanceof PsiJavaFile) {
      final PsiJavaFile javaFile = (PsiJavaFile)file;
      PsiClass[] classes = javaFile.getClasses();
      if (classes.length == 1 && classes[0] instanceof PsiImplicitClass) {
        return (PsiImplicitClass)classes[0];
      }
    }
    return null;
  }

  /**
   * @param name the name of the implicitly declared class (might include the ".java" extension)
   * @return the JVM name of the implicitly declared class
   */
  public static String getJvmName(@NotNull String name) {
    return StringUtil.trimEnd(name, ".java", true);
  }
}
