/*
 * Copyright 2010-2020 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis

import ksp.org.jetbrains.kotlin.fir.FirSession
import ksp.org.jetbrains.kotlin.fir.FirSessionComponent
import ksp.org.jetbrains.kotlin.fir.NoMutableState
import ksp.org.jetbrains.kotlin.fir.SessionConfiguration
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.LanguageVersionSettingsCheckers
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.config.ComposedLanguageVersionSettingsCheckers
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.declaration.ComposedDeclarationCheckers
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.declaration.DeclarationCheckers
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.expression.ComposedExpressionCheckers
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.expression.ExpressionCheckers
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.type.ComposedTypeCheckers
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.type.TypeCheckers
import ksp.org.jetbrains.kotlin.fir.analysis.extensions.FirAdditionalCheckersExtension

@RequiresOptIn
annotation class CheckersComponentInternal

/**
 * [CheckersComponent] stores various kinds of checkers in the CLI compiler mode. In the Analysis API mode, this component is not
 * registered, as `LLCheckersFactory` is used instead.
 */
@NoMutableState
class CheckersComponent : FirSessionComponent {
    val commonDeclarationCheckers: DeclarationCheckers get() = _commonDeclarationCheckers
    val platformDeclarationCheckers: DeclarationCheckers get() = _platformDeclarationCheckers

    private val _commonDeclarationCheckers = ComposedDeclarationCheckers(MppCheckerKind.Common)
    private val _platformDeclarationCheckers = ComposedDeclarationCheckers(MppCheckerKind.Platform)

    val commonExpressionCheckers: ExpressionCheckers get() = _commonExpressionCheckers
    val platformExpressionCheckers: ExpressionCheckers get() = _platformExpressionCheckers

    private val _commonExpressionCheckers = ComposedExpressionCheckers(MppCheckerKind.Common)
    private val _platformExpressionCheckers = ComposedExpressionCheckers(MppCheckerKind.Platform)

    val commonTypeCheckers: TypeCheckers get() = _commonTypeCheckers
    val platformTypeCheckers: TypeCheckers get() = _platformTypeCheckers

    private val _commonTypeCheckers = ComposedTypeCheckers(MppCheckerKind.Common)
    private val _platformTypeCheckers = ComposedTypeCheckers(MppCheckerKind.Platform)

    val languageVersionSettingsCheckers: LanguageVersionSettingsCheckers get() = _languageVersionSettingsCheckers
    private val _languageVersionSettingsCheckers = ComposedLanguageVersionSettingsCheckers()

    @SessionConfiguration
    @OptIn(CheckersComponentInternal::class)
    fun register(checkers: DeclarationCheckers) {
        _commonDeclarationCheckers.register(checkers)
        _platformDeclarationCheckers.register(checkers)
    }

    @SessionConfiguration
    @OptIn(CheckersComponentInternal::class)
    fun register(checkers: ExpressionCheckers) {
        _commonExpressionCheckers.register(checkers)
        _platformExpressionCheckers.register(checkers)
    }

    @SessionConfiguration
    @OptIn(CheckersComponentInternal::class)
    fun register(checkers: TypeCheckers) {
        _commonTypeCheckers.register(checkers)
        _platformTypeCheckers.register(checkers)
    }

    @SessionConfiguration
    @OptIn(CheckersComponentInternal::class)
    fun register(checkers: LanguageVersionSettingsCheckers) {
        _languageVersionSettingsCheckers.register(checkers)
    }

    @SessionConfiguration
    fun register(checkers: FirAdditionalCheckersExtension) {
        register(checkers.declarationCheckers)
        register(checkers.expressionCheckers)
        register(checkers.typeCheckers)
        register(checkers.languageVersionSettingsCheckers)
    }
}

val FirSession.nullableCheckersComponent: CheckersComponent? by FirSession.nullableSessionComponentAccessor()

val FirSession.checkersComponent: CheckersComponent
    get() = nullableCheckersComponent ?: error("Expected `${CheckersComponent::class}` to be registered in CLI compiler mode.")
