/*
 * Copyright 2010-2025 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.checkers.type

import ksp.org.jetbrains.kotlin.KtRealSourceElementKind
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.SourceNavigator
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.toClassLikeSymbol
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.FirErrors
import ksp.org.jetbrains.kotlin.fir.resolve.toSymbol
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirTypeAliasSymbol
import ksp.org.jetbrains.kotlin.fir.types.FirResolvedTypeRef
import ksp.org.jetbrains.kotlin.fir.types.abbreviatedTypeOrSelf
import ksp.org.jetbrains.kotlin.fir.types.isMarkedNullable

object RedundantNullableChecker : FirResolvedTypeRefChecker(MppCheckerKind.Common) {
    context(context: CheckerContext, reporter: DiagnosticReporter)
    override fun check(typeRef: FirResolvedTypeRef) {
        if (
            !typeRef.coneType.abbreviatedTypeOrSelf.isMarkedNullable ||
            typeRef.source?.kind !is KtRealSourceElementKind
        ) return

        var symbol = typeRef.coneType.abbreviatedTypeOrSelf.toSymbol()
        if (symbol is FirTypeAliasSymbol) {
            while (symbol is FirTypeAliasSymbol) {
                val resolvedExpandedTypeRef = symbol.resolvedExpandedTypeRef
                if (resolvedExpandedTypeRef.coneType.isMarkedNullable) {
                    reporter.reportOn(typeRef.source, FirErrors.REDUNDANT_NULLABLE)
                    break
                } else {
                    symbol = resolvedExpandedTypeRef.toClassLikeSymbol(context.session)
                }
            }
        } else {
            with(SourceNavigator.forElement(typeRef)) {
                if (typeRef.isRedundantNullable()) {
                    reporter.reportOn(typeRef.source, FirErrors.REDUNDANT_NULLABLE)
                }
            }
        }
    }
}