/*
 * Copyright 2010-2020 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.ir.backend.js.lower

import ksp.org.jetbrains.kotlin.backend.common.lower.DefaultParameterInjector
import ksp.org.jetbrains.kotlin.ir.UNDEFINED_OFFSET
import ksp.org.jetbrains.kotlin.ir.backend.js.JsIrBackendContext
import ksp.org.jetbrains.kotlin.ir.backend.js.JsLoweredDeclarationOrigin
import ksp.org.jetbrains.kotlin.ir.backend.js.JsStatementOrigins
import ksp.org.jetbrains.kotlin.ir.backend.js.lower.coroutines.PrepareSuspendFunctionsForExportLowering.Companion.promisifiedWrapperFunction
import ksp.org.jetbrains.kotlin.ir.backend.js.tsexport.isExported
import ksp.org.jetbrains.kotlin.ir.backend.js.utils.getVoid
import ksp.org.jetbrains.kotlin.ir.backend.js.utils.jsConstructorReference
import ksp.org.jetbrains.kotlin.ir.builders.IrBlockBuilder
import ksp.org.jetbrains.kotlin.ir.declarations.IrFunction
import ksp.org.jetbrains.kotlin.ir.declarations.IrValueParameter
import ksp.org.jetbrains.kotlin.ir.expressions.IrCall
import ksp.org.jetbrains.kotlin.ir.expressions.IrExpression
import ksp.org.jetbrains.kotlin.ir.expressions.IrFunctionAccessExpression
import ksp.org.jetbrains.kotlin.ir.expressions.impl.IrCallImpl
import ksp.org.jetbrains.kotlin.ir.symbols.IrClassSymbol
import ksp.org.jetbrains.kotlin.ir.util.isTopLevel
import ksp.org.jetbrains.kotlin.ir.util.isVararg
import ksp.org.jetbrains.kotlin.ir.util.nonDispatchArguments

class JsDefaultParameterInjector(context: JsIrBackendContext) :
    DefaultParameterInjector<JsIrBackendContext>(
        context,
        factory = JsDefaultArgumentFunctionFactory(context),
        skipExternalMethods = true,
        forceSetOverrideSymbols = false
    ) {
    override fun nullConst(startOffset: Int, endOffset: Int, irParameter: IrValueParameter): IrExpression? =
        if (irParameter.isVararg && !irParameter.hasDefaultValue()) {
            null
        } else {
            context.getVoid()
        }

    override fun shouldReplaceWithSyntheticFunction(functionAccess: IrFunctionAccessExpression): Boolean {
        return functionAccess.nonDispatchArguments.any { it == null } || functionAccess.symbol.owner.run {
            origin == JsLoweredDeclarationOrigin.JS_SHADOWED_EXPORT &&
                    !isTopLevel &&
                    functionAccess.origin != JsStatementOrigins.IMPLEMENTATION_DELEGATION_CALL &&
                    (promisifiedWrapperFunction ?: this).isExported(context)
        }
    }

    override fun IrBlockBuilder.argumentsForCall(
        expression: IrFunctionAccessExpression,
        stubFunction: IrFunction,
    ): Map<IrValueParameter, IrExpression?> {
        val startOffset = expression.startOffset
        val endOffset = expression.endOffset

        return buildMap {
            for (i in 0..<expression.arguments.size) {
                val parameter = stubFunction.parameters[i]
                val argument = expression.arguments[i]
                put(parameter, argument ?: nullConst(startOffset, endOffset, parameter))
            }

            if (expression is IrCall && stubFunction.hasSuperContextParameter()) {
                put(
                    stubFunction.parameters[expression.arguments.size],
                    expression.superQualifierSymbol?.prototypeOf() ?: this@JsDefaultParameterInjector.context.getVoid()
                )
            }
        }
    }

    private fun IrFunction.hasSuperContextParameter(): Boolean =
        parameters.lastOrNull()?.origin == JsLoweredDeclarationOrigin.JS_SUPER_CONTEXT_PARAMETER

    private fun IrClassSymbol.prototypeOf(): IrExpression {
        return IrCallImpl(
            UNDEFINED_OFFSET,
            UNDEFINED_OFFSET,
            context.dynamicType,
            context.intrinsics.jsPrototypeOfSymbol,
            typeArgumentsCount = 0,
        ).apply {
            arguments[0] = owner.jsConstructorReference(context)
        }
    }

    private fun IrValueParameter.hasDefaultValue(): Boolean =
        origin == JsLoweredDeclarationOrigin.JS_SHADOWED_DEFAULT_PARAMETER
}

