/*
 * Copyright 2010-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package ksp.org.jetbrains.kotlin.types;

import ksp.org.jetbrains.annotations.NotNull;
import ksp.org.jetbrains.kotlin.descriptors.ClassDescriptor;
import ksp.org.jetbrains.kotlin.descriptors.SupertypeLoopChecker;
import ksp.org.jetbrains.kotlin.descriptors.TypeParameterDescriptor;
import ksp.org.jetbrains.kotlin.resolve.DescriptorUtils;
import ksp.org.jetbrains.kotlin.storage.LockBasedStorageManager;
import ksp.org.jetbrains.kotlin.storage.StorageManager;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

public class ClassTypeConstructorImpl extends AbstractClassTypeConstructor implements TypeConstructor {
    private final ClassDescriptor classDescriptor;
    private final List<TypeParameterDescriptor> parameters;
    private final Collection<KotlinType> supertypes;

    public ClassTypeConstructorImpl(
            @NotNull ClassDescriptor classDescriptor,
            @NotNull List<? extends TypeParameterDescriptor> parameters,
            @NotNull Collection<KotlinType> supertypes,
            @NotNull StorageManager storageManager
    ) {
        super(storageManager);
        this.classDescriptor = classDescriptor;
        this.parameters = Collections.unmodifiableList(new ArrayList<TypeParameterDescriptor>(parameters));
        this.supertypes = Collections.unmodifiableCollection(supertypes);
    }

    @Override
    @NotNull
    public List<TypeParameterDescriptor> getParameters() {
        return parameters;
    }

    @Override
    public String toString() {
        return DescriptorUtils.getFqName(classDescriptor).asString();
    }

    @Override
    public boolean isDenotable() {
        return true;
    }

    @Override
    @NotNull
    public ClassDescriptor getDeclarationDescriptor() {
        return classDescriptor;
    }

    @NotNull
    @Override
    protected Collection<KotlinType> computeSupertypes() {
        return supertypes;
    }

    @NotNull
    @Override
    protected SupertypeLoopChecker getSupertypeLoopChecker() {
        return SupertypeLoopChecker.EMPTY.INSTANCE;
    }
}
