/*
 * Copyright 2010-2018 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.backend.common.lower

import ksp.org.jetbrains.kotlin.backend.common.BodyLoweringPass
import ksp.org.jetbrains.kotlin.backend.common.DeclarationTransformer
import ksp.org.jetbrains.kotlin.ir.IrStatement
import ksp.org.jetbrains.kotlin.ir.declarations.*
import ksp.org.jetbrains.kotlin.ir.expressions.IrBody
import ksp.org.jetbrains.kotlin.ir.expressions.impl.IrBlockImpl
import ksp.org.jetbrains.kotlin.ir.util.isEffectivelyExternal
import ksp.org.jetbrains.kotlin.ir.visitors.*

/**
 * Moves fields and accessors out from its property.
 */
class PropertiesLowering : DeclarationTransformer {
    override val withLocalDeclarations: Boolean get() = true

    override fun transformFlat(declaration: IrDeclaration): List<IrDeclaration>? {
        when (declaration) {
            is IrSimpleFunction -> {
                declaration.correspondingPropertySymbol?.owner?.let { property ->
                    if (!property.isEffectivelyExternal()) {
                        return listOf(declaration)
                    }
                }
            }
            is IrField -> {
                declaration.correspondingPropertySymbol?.owner?.let { property ->
                    if (!property.isEffectivelyExternal()) {
                        return listOf(declaration)
                    }
                }
            }
            is IrProperty -> {
                if (!declaration.isEffectivelyExternal()) {
                    return listOf()
                }
            }
        }

        return null
    }
}

class LocalDelegatedPropertiesLowering : IrElementTransformerVoid(), BodyLoweringPass {

    override fun lower(irBody: IrBody, container: IrDeclaration) {
        irBody.accept(this, null)
    }

    override fun visitLocalDelegatedProperty(declaration: IrLocalDelegatedProperty): IrStatement {
        declaration.transformChildrenVoid(this)

        val initializer = declaration.delegate.initializer!!
        declaration.delegate.initializer = IrBlockImpl(
            initializer.startOffset, initializer.endOffset, initializer.type, null,
            listOfNotNull(
                declaration.getter,
                declaration.setter,
                initializer
            )
        )

        return declaration.delegate
    }
}
