/*
 * Copyright 2010-2021 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.checkers.expression

import ksp.org.jetbrains.kotlin.KtNodeTypes
import ksp.org.jetbrains.kotlin.KtSourceElement
import ksp.org.jetbrains.kotlin.config.LanguageFeature
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.FirErrors
import ksp.org.jetbrains.kotlin.fir.expressions.FirSafeCallExpression
import ksp.org.jetbrains.kotlin.fir.isEnabled
import ksp.org.jetbrains.kotlin.fir.resolve.fullyExpandedType
import ksp.org.jetbrains.kotlin.fir.types.ConeKotlinType
import ksp.org.jetbrains.kotlin.fir.types.canBeNull
import ksp.org.jetbrains.kotlin.fir.types.resolvedType

abstract class AbstractFirUnnecessarySafeCallChecker : FirSafeCallExpressionChecker(MppCheckerKind.Common) {
    context(context: CheckerContext, reporter: DiagnosticReporter)
    protected fun checkSafeCallReceiverType(
        receiverType: ConeKotlinType,
        source: KtSourceElement?,
    ) {
        if (!receiverType.canBeNull(context.session)) {
            if (LanguageFeature.EnableDfaWarningsInK2.isEnabled()) {
                reporter.reportOn(source, FirErrors.UNNECESSARY_SAFE_CALL, receiverType)
            }
        }
    }
}

object FirUnnecessarySafeCallChecker : AbstractFirUnnecessarySafeCallChecker() {
    context(context: CheckerContext, reporter: DiagnosticReporter)
    override fun check(expression: FirSafeCallExpression) {
        val receiverType = expression.receiver.resolvedType.fullyExpandedType()
        if (expression.receiver.source?.elementType == KtNodeTypes.SUPER_EXPRESSION) {
            reporter.reportOn(expression.source, FirErrors.UNEXPECTED_SAFE_CALL)
            return
        }
        checkSafeCallReceiverType(receiverType, expression.source)
    }
}
