// Copyright 2000-2023 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license.
package ksp.com.intellij.openapi.util;

import ksp.com.intellij.util.containers.ContainerUtil;
import ksp.com.intellij.util.xmlb.Constants;
import ksp.org.jdom.Element;
import ksp.org.jetbrains.annotations.ApiStatus;
import ksp.org.jetbrains.annotations.NonNls;
import ksp.org.jetbrains.annotations.NotNull;
import ksp.org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.List;

public final class JDOMExternalizerUtil {
  /**
   * Adds the {@code <option name="{fieldName}" value="{value}"/>} element to the parent.
   */
  public static void writeField(@NotNull Element parent, @NotNull @NonNls String fieldName, @Nullable String value) {
    Element element = new Element(Constants.OPTION);
    element.setAttribute(Constants.NAME, fieldName);
    element.setAttribute(Constants.VALUE, value == null ? "" : value);
    parent.addContent(element);
  }

  /**
   * Adds the {@code <option name="{fieldName}" value="{value}"/>} element to the parent when the value differs from the default.
   */
  public static void writeField(@NotNull Element parent, @NotNull @NonNls String fieldName, @Nullable String value, @NotNull String defaultValue) {
    if (!defaultValue.equals(value)) {
      writeField(parent, fieldName, value);
    }
  }

  public static @NotNull String readField(@NotNull Element parent, @NotNull @NonNls String fieldName, @NotNull String defaultValue) {
    String val = readField(parent, fieldName);
    return val == null ? defaultValue : val;
  }

  public static @Nullable String readField(@NotNull Element parent, @NonNls @NotNull String fieldName) {
    for (Element element : parent.getChildren(Constants.OPTION)) {
      if (fieldName.equals(element.getAttributeValue(Constants.NAME))) {
        return element.getAttributeValue(Constants.VALUE);
      }
    }
    return null;
  }

  /**
   * Adds the {@code <option name="{fieldName}"/>} element to the parent and returns the created element.
   */
  public static @NotNull Element writeOption(@NotNull Element parent, @NotNull String fieldName) {
    Element element = new Element(Constants.OPTION);
    element.setAttribute(Constants.NAME, fieldName);
    parent.addContent(element);
    return element;
  }

  public static @Nullable Element readOption(@NotNull Element parent, @NotNull String fieldName) {
    for (Element element : parent.getChildren(Constants.OPTION)) {
      if (fieldName.equals(element.getAttributeValue(Constants.NAME))) {
        return element;
      }
    }
    return null;
  }

  /**
   * Adds the {@code <{tagName} value="{value}"/>} element to the parent (or just {@code <{tagName}"/>} if the value is {@code null}).
   */
  public static void writeCustomField(@NotNull Element parent, @NotNull String tagName, @Nullable String value) {
    Element element = new Element(tagName);
    if (value != null) {
      element.setAttribute(Constants.VALUE, value);
    }
    parent.addContent(element);
  }

  public static @Nullable String readCustomField(@NotNull Element parent, @NotNull String tagName) {
    Element element = parent.getChild(tagName);
    return element != null ? element.getAttributeValue(Constants.VALUE) : null;
  }

  public static @NotNull List<String> getChildrenValueAttributes(@NotNull Element parent, @NotNull String childTagName) {
    return ContainerUtil.mapNotNull(parent.getChildren(childTagName), e -> e.getAttributeValue(Constants.VALUE));
  }

  public static void addChildrenWithValueAttribute(@NotNull Element parent,
                                                   @NotNull String childTagName,
                                                   @NotNull List<String> attrValues) {
    for (String value : attrValues) {
      if (value != null) {
        Element child = new Element(childTagName);
        child.setAttribute(Constants.VALUE, value);
        parent.addContent(child);
      }
    }
  }

  @SuppressWarnings("deprecation")
  public static void addChildren(@NotNull Element parent,
                                 @NotNull String childElementName,
                                 @NotNull Collection<? extends JDOMExternalizable> children) throws WriteExternalException {
    for (JDOMExternalizable child : children) {
      if (child != null) {
        Element element = new Element(childElementName);
        child.writeExternal(element);
        parent.addContent(element);
      }
    }
  }

  //<editor-fold desc="Deprecated stuff.">
  /** @deprecated use {@link #writeCustomField(Element, String, String)} */
  @Deprecated
  @ApiStatus.ScheduledForRemoval
  public static Element addElementWithValueAttribute(@NotNull Element parent, @NotNull String childTagName, @Nullable String attrValue) {
    writeCustomField(parent, childTagName, attrValue);
    return parent.getChild(childTagName);
  }

  /** @deprecated use {@link #readCustomField(Element, String)} */
  @Deprecated
  @ApiStatus.ScheduledForRemoval
  public static String getFirstChildValueAttribute(@NotNull Element parent, @NotNull String childTagName) {
    return readCustomField(parent, childTagName);
  }
  //</editor-fold>
}
