/*
 * Copyright 2010-2024 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.analysis.api.fir.symbols

import ksp.com.intellij.psi.PsiElement
import ksp.org.jetbrains.kotlin.KtFakeSourceElementKind
import ksp.org.jetbrains.kotlin.analysis.api.annotations.KaAnnotationList
import ksp.org.jetbrains.kotlin.analysis.api.fir.KaFirSession
import ksp.org.jetbrains.kotlin.analysis.api.fir.findPsi
import ksp.org.jetbrains.kotlin.analysis.api.fir.parameterName
import ksp.org.jetbrains.kotlin.analysis.api.fir.symbols.pointers.KaFirValueParameterSymbolPointer
import ksp.org.jetbrains.kotlin.analysis.api.fir.symbols.pointers.createOwnerPointer
import ksp.org.jetbrains.kotlin.analysis.api.fir.utils.firSymbol
import ksp.org.jetbrains.kotlin.analysis.api.lifetime.withValidityAssertion
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaFunctionSymbol
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaKotlinPropertySymbol
import ksp.org.jetbrains.kotlin.analysis.api.symbols.KaValueParameterSymbol
import ksp.org.jetbrains.kotlin.analysis.api.symbols.pointers.KaSymbolPointer
import ksp.org.jetbrains.kotlin.analysis.api.types.KaType
import ksp.org.jetbrains.kotlin.analysis.utils.errors.requireIsInstance
import ksp.org.jetbrains.kotlin.descriptors.Visibility
import ksp.org.jetbrains.kotlin.fir.correspondingProperty
import ksp.org.jetbrains.kotlin.fir.declarations.FirFunction
import ksp.org.jetbrains.kotlin.fir.declarations.impl.FirResolvedDeclarationStatusImpl
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirValueParameterSymbol
import ksp.org.jetbrains.kotlin.fir.types.varargElementType
import ksp.org.jetbrains.kotlin.lexer.KtTokens
import ksp.org.jetbrains.kotlin.name.Name
import ksp.org.jetbrains.kotlin.psi.KtParameter
import ksp.org.jetbrains.kotlin.psi.KtPrimaryConstructor

internal class KaFirValueParameterSymbol private constructor(
    override val backingPsi: KtParameter?,
    override val analysisSession: KaFirSession,
    override val lazyFirSymbol: Lazy<FirValueParameterSymbol>,
) : KaValueParameterSymbol(), KaFirKtBasedSymbol<KtParameter, FirValueParameterSymbol> {
    constructor(declaration: KtParameter, session: KaFirSession) : this(
        backingPsi = declaration,
        lazyFirSymbol = lazyFirSymbol(declaration, session),
        analysisSession = session,
    )

    constructor(symbol: FirValueParameterSymbol, session: KaFirSession) : this(
        backingPsi = symbol.backingPsiIfApplicable as? KtParameter,
        lazyFirSymbol = lazyOf(symbol),
        analysisSession = session,
    )

    override val psi: PsiElement?
        get() = withValidityAssertion { backingPsi ?: findPsi() }

    override val name: Name
        get() = withValidityAssertion { backingPsi?.parameterName ?: firSymbol.name }

    override val isVararg: Boolean
        get() = withValidityAssertion { backingPsi?.isVarArg ?: firSymbol.isVararg }

    override val isImplicitLambdaParameter: Boolean
        get() = withValidityAssertion {
            if (backingPsi != null)
                false
            else
                firSymbol.source?.kind == KtFakeSourceElementKind.ItLambdaParameter
        }

    override val isCrossinline: Boolean
        get() = withValidityAssertion { backingPsi?.hasModifier(KtTokens.CROSSINLINE_KEYWORD) ?: firSymbol.isCrossinline }

    override val compilerVisibility: Visibility
        get() = withValidityAssertion { FirResolvedDeclarationStatusImpl.DEFAULT_STATUS_FOR_STATUSLESS_DECLARATIONS.visibility }

    override val isNoinline: Boolean
        get() = withValidityAssertion { backingPsi?.hasModifier(KtTokens.NOINLINE_KEYWORD) ?: firSymbol.isNoinline }

    override val returnType: KaType
        get() = withValidityAssertion {
            val returnType = firSymbol.resolvedReturnType
            if (firSymbol.isVararg) {
                builder.typeBuilder.buildKtType(returnType.varargElementType())
            } else {
                builder.typeBuilder.buildKtType(returnType)
            }
        }

    override val hasDefaultValue: Boolean
        get() = withValidityAssertion { backingPsi?.hasDefaultValue() ?: firSymbol.hasDefaultValue }

    override val annotations: KaAnnotationList
        get() = withValidityAssertion { psiOrSymbolAnnotationList() }

    override val generatedPrimaryConstructorProperty: KaKotlinPropertySymbol?
        get() = withValidityAssertion {
            if (backingPsi != null) {
                return if (backingPsi.hasValOrVar() && backingPsi.ownerFunction is KtPrimaryConstructor) {
                    KaFirKotlinPropertySymbol.create(backingPsi, analysisSession)
                } else {
                    null
                }
            }

            val propertySymbol = firSymbol.fir.correspondingProperty?.symbol ?: return null
            return KaFirKotlinPropertySymbol.create(propertySymbol, analysisSession)
        }

    override fun createPointer(): KaSymbolPointer<KaValueParameterSymbol> = withValidityAssertion {
        psiBasedSymbolPointerOfTypeIfSource<KaValueParameterSymbol>()?.let { return it }

        val ownerSymbol = with(analysisSession) { containingDeclaration }
            ?: error("Containing function is expected for a value parameter symbol")

        requireIsInstance<KaFunctionSymbol>(ownerSymbol)

        return KaFirValueParameterSymbolPointer(
            ownerPointer = analysisSession.createOwnerPointer(this),
            name = name,
            index = (ownerSymbol.firSymbol.fir as FirFunction).valueParameters.indexOf(firSymbol.fir),
            originalSymbol = this
        )
    }

    override fun equals(other: Any?): Boolean {
        return psiOrSymbolEquals(other)
    }

    override fun hashCode(): Int {
        return psiOrSymbolHashCode()
    }
}
