/*
 * Copyright 2010-2024 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.checkers.experimental

import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.expression.FirLiteralExpressionChecker
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.expression.FirStringConcatenationCallChecker
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.FirErrors
import ksp.org.jetbrains.kotlin.fir.expressions.FirLiteralExpression
import ksp.org.jetbrains.kotlin.fir.expressions.FirStringConcatenationCall
import ksp.org.jetbrains.kotlin.text
import ksp.org.jetbrains.kotlin.types.ConstantValueKind

object RedundantInterpolationPrefixCheckerConcatenation : FirStringConcatenationCallChecker(MppCheckerKind.Common) {
    context(context: CheckerContext, reporter: DiagnosticReporter)
    override fun check(expression: FirStringConcatenationCall) {
        if (expression.interpolationPrefix == "$") {
            reporter.reportOn(expression.source, FirErrors.REDUNDANT_INTERPOLATION_PREFIX)
        }
    }
}

object RedundantInterpolationPrefixCheckerLiteral : FirLiteralExpressionChecker(MppCheckerKind.Common) {
    context(context: CheckerContext, reporter: DiagnosticReporter)
    override fun check(expression: FirLiteralExpression) {
        val prefix = expression.prefix
        if (expression.kind == ConstantValueKind.String && !prefix.isNullOrEmpty()) {
            val value = expression.source.text?.drop(prefix.length) ?: return
            // approximation of interpolated values: $ followed either by start of an identifier, or braces
            if (!Regex("""[^\\]\$(\w|\{|`[^`])""").containsMatchIn(value)) {
                reporter.reportOn(expression.source, FirErrors.REDUNDANT_INTERPOLATION_PREFIX)
            }
        }
    }
}
