/*
 * Copyright 2010-2023 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.js.checkers

import ksp.org.jetbrains.kotlin.AbstractKtSourceElement
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.fir.FirSession
import ksp.org.jetbrains.kotlin.fir.FirSessionComponent
import ksp.org.jetbrains.kotlin.fir.NoMutableState
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.getAnnotationStringParameter
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.js.FirJsErrors
import ksp.org.jetbrains.kotlin.fir.declarations.hasAnnotation
import ksp.org.jetbrains.kotlin.fir.symbols.FirBasedSymbol
import ksp.org.jetbrains.kotlin.name.JsStandardClassIds
import ksp.org.jetbrains.kotlin.serialization.js.ModuleKind

@NoMutableState
class FirJsModuleKind(val moduleKind: ModuleKind) : FirSessionComponent

private val FirSession.jsModuleKindComponent: FirJsModuleKind? by FirSession.nullableSessionComponentAccessor()

private val FirSession.jsModuleKind: ModuleKind?
    get() = jsModuleKindComponent?.moduleKind

context(context: CheckerContext, reporter: DiagnosticReporter)
internal fun checkJsModuleUsage(
    callee: FirBasedSymbol<*>,
    source: AbstractKtSourceElement?
) {
    val moduleKind = context.session.jsModuleKind ?: return

    val calleeSession = callee.moduleData.session
    val calleeRoot = getRootClassLikeSymbolOrSelf(callee, calleeSession)
    val calleeContainingFile = calleeRoot.getContainingFile()

    val callToModule = calleeRoot.getAnnotationStringParameter(JsStandardClassIds.Annotations.JsModule, calleeSession) != null ||
            calleeContainingFile?.symbol?.getAnnotationStringParameter(JsStandardClassIds.Annotations.JsModule, calleeSession) != null

    val callToNonModule = calleeRoot.hasAnnotation(JsStandardClassIds.Annotations.JsNonModule, calleeSession) ||
            calleeContainingFile?.symbol?.hasAnnotation(JsStandardClassIds.Annotations.JsNonModule, calleeSession) == true

    when (moduleKind) {
        ModuleKind.UMD -> {
            if (!callToNonModule && callToModule || callToNonModule && !callToModule) {
                reporter.reportOn(source, FirJsErrors.CALL_FROM_UMD_MUST_BE_JS_MODULE_AND_JS_NON_MODULE)
            }
        }
        ModuleKind.PLAIN -> {
            if (!callToNonModule && callToModule) {
                reporter.reportOn(source, FirJsErrors.CALL_TO_JS_MODULE_WITHOUT_MODULE_SYSTEM, callee)
            }
        }
        else -> {
            if (!callToModule && callToNonModule) {
                reporter.reportOn(source, FirJsErrors.CALL_TO_JS_NON_MODULE_WITH_MODULE_SYSTEM, callee)
            }
        }
    }
}
