/*
 * Copyright 2010-2021 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.checkers.syntax

import ksp.com.intellij.lang.LighterASTNode
import ksp.com.intellij.psi.PsiElement
import ksp.com.intellij.psi.tree.IElementType
import ksp.com.intellij.util.diff.FlyweightCapableTreeStructure
import ksp.org.jetbrains.kotlin.*
import ksp.org.jetbrains.kotlin.KtNodeTypes.BINARY_EXPRESSION
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.FirErrors
import ksp.org.jetbrains.kotlin.fir.expressions.FirStatement
import ksp.org.jetbrains.kotlin.lexer.KtKeywordToken
import ksp.org.jetbrains.kotlin.lexer.KtTokens
import ksp.org.jetbrains.kotlin.psi.KtExpression
import ksp.org.jetbrains.kotlin.psi.psiUtil.nextLeaf
import ksp.org.jetbrains.kotlin.psi.psiUtil.prevLeaf
import ksp.org.jetbrains.kotlin.util.getChildren

object FirPrefixAndSuffixSyntaxChecker : FirExpressionSyntaxChecker<FirStatement, KtExpression>() {

    private val literalConstants = listOf(KtNodeTypes.CHARACTER_CONSTANT, KtNodeTypes.FLOAT_CONSTANT, KtNodeTypes.INTEGER_CONSTANT)

    override fun isApplicable(element: FirStatement, source: KtSourceElement): Boolean =
        source.kind !is KtFakeSourceElementKind && (source.elementType == KtNodeTypes.STRING_TEMPLATE || source.elementType in literalConstants)

    context(context: CheckerContext, reporter: DiagnosticReporter)
    override fun checkPsi(
        element: FirStatement,
        source: KtPsiSourceElement,
        psi: KtExpression,
    ) {
        psi.prevLeaf()?.let { checkLiteralPrefixOrSuffix(it) }
        psi.nextLeaf()?.let { checkLiteralPrefixOrSuffix(it) }
    }


    context(context: CheckerContext, reporter: DiagnosticReporter)
    override fun checkLightTree(
        element: FirStatement,
        source: KtLightSourceElement,
    ) {
        source.lighterASTNode.prevLeaf(source.treeStructure)
            ?.let { checkLiteralPrefixOrSuffix(it, source) }
        source.lighterASTNode.nextLeaf(source.treeStructure)
            ?.let { checkLiteralPrefixOrSuffix(it, source) }
    }

    private enum class Direction(val offset: Int) {
        PREVIOUS(-1),
        NEXT(1)
    }

    private fun LighterASTNode.getLeaf(
        direction: Direction,
        treeStructure: FlyweightCapableTreeStructure<LighterASTNode>,
    ): LighterASTNode? {
        val parent = treeStructure.getParent(this) ?: return null
        val children = parent.getChildren(treeStructure)
        val index = children.indexOf(this)
        val leaf = children.getOrNull(index - direction.offset)
        return when {
            // Necessary for finding the next leaf in complex binary expressions, for example 'a foo"asdsfsa"foo a'
            leaf == null && parent.tokenType == BINARY_EXPRESSION -> parent.getLeaf(direction, treeStructure)
            leaf == null -> return null
            else -> {
                // This is necessary to obtain the simplest node, as the found leaf can be a complex expression
                var result = leaf
                var resultChildren = leaf.getChildren(treeStructure)
                while (resultChildren.isNotEmpty()) {
                    result = if (direction == Direction.PREVIOUS) resultChildren.first() else resultChildren.last()
                    resultChildren = result.getChildren(treeStructure)
                }
                result
            }
        }
    }

    private fun LighterASTNode.prevLeaf(treeStructure: FlyweightCapableTreeStructure<LighterASTNode>): LighterASTNode? {
        return getLeaf(Direction.PREVIOUS, treeStructure)
    }

    private fun LighterASTNode.nextLeaf(treeStructure: FlyweightCapableTreeStructure<LighterASTNode>): LighterASTNode? {
        return getLeaf(Direction.NEXT, treeStructure)
    }

    context(context: CheckerContext, reporter: DiagnosticReporter)
    private fun checkLiteralPrefixOrSuffix(
        prefixOrSuffix: PsiElement,
    ) {
        if (illegalLiteralPrefixOrSuffix(prefixOrSuffix.node.elementType)) {
            report(prefixOrSuffix.toKtPsiSourceElement())
        }
    }

    context(context: CheckerContext, reporter: DiagnosticReporter)
    private fun checkLiteralPrefixOrSuffix(
        prefixOrSuffix: LighterASTNode,
        source: KtSourceElement,
    ) {
        val elementType = prefixOrSuffix.tokenType ?: return
        if (illegalLiteralPrefixOrSuffix(elementType)) {
            report(prefixOrSuffix.toKtLightSourceElement(source.treeStructure))
        }
    }

    private fun illegalLiteralPrefixOrSuffix(elementType: IElementType): Boolean =
        (elementType === KtTokens.IDENTIFIER || elementType === KtTokens.INTEGER_LITERAL || elementType === KtTokens.FLOAT_LITERAL || elementType is KtKeywordToken)


    context(context: CheckerContext, reporter: DiagnosticReporter)
    private fun report(source: KtSourceElement) {
        reporter.reportOn(source, FirErrors.UNSUPPORTED, "Literals must be surrounded by whitespace.")
    }
}
