/*
 * Copyright 2010-2019 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.java.scopes

import ksp.org.jetbrains.kotlin.fir.FirSession
import ksp.org.jetbrains.kotlin.fir.declarations.FirCallableDeclaration
import ksp.org.jetbrains.kotlin.fir.declarations.FirDeclarationOrigin
import ksp.org.jetbrains.kotlin.fir.declarations.FirSimpleFunction
import ksp.org.jetbrains.kotlin.fir.initialSignatureAttr
import ksp.org.jetbrains.kotlin.fir.java.enhancement.FirSignatureEnhancement
import ksp.org.jetbrains.kotlin.fir.resolve.ScopeSession
import ksp.org.jetbrains.kotlin.fir.scopes.*
import ksp.org.jetbrains.kotlin.fir.symbols.impl.*
import ksp.org.jetbrains.kotlin.name.Name

class JavaClassMembersEnhancementScope(
    session: FirSession,
    private val owner: FirRegularClassSymbol,
    private val useSiteMemberScope: JavaClassUseSiteMemberScope,
) : FirDelegatingTypeScope(useSiteMemberScope) {
    private val enhancedToOriginalFunctions = hashMapOf<FirNamedFunctionSymbol, FirNamedFunctionSymbol>()
    private val enhancedToOriginalProperties = hashMapOf<FirPropertySymbol, FirPropertySymbol>()

    private val signatureEnhancement = FirSignatureEnhancement(owner.fir, session) {
        overriddenMembers()
    }

    override fun processPropertiesByName(name: Name, processor: (FirVariableSymbol<*>) -> Unit) {
        useSiteMemberScope.processPropertiesByName(name) process@{ original ->
            val enhancedPropertySymbol = signatureEnhancement.enhancedProperty(original, name)
            if (original is FirPropertySymbol && enhancedPropertySymbol is FirPropertySymbol) {
                enhancedToOriginalProperties[enhancedPropertySymbol] = original
            }

            processor(enhancedPropertySymbol)
        }
    }

    override fun processFunctionsByName(name: Name, processor: (FirNamedFunctionSymbol) -> Unit) {
        useSiteMemberScope.processFunctionsByName(name) process@{ original ->
            val symbol = signatureEnhancement.enhancedFunction(original, name)
            val enhancedFunction = symbol.fir
            val enhancedFunctionSymbol = enhancedFunction.symbol
            enhancedToOriginalFunctions[enhancedFunctionSymbol] = original
            processor(enhancedFunctionSymbol)
        }
    }

    @OptIn(ScopeFunctionRequiresPrewarm::class)
    private fun FirCallableDeclaration.overriddenMembers(): List<FirCallableDeclaration> {
        return useSiteMemberScope.getDirectOverriddenMembers(symbol).map { it.fir }
    }

    override fun processDeclaredConstructors(processor: (FirConstructorSymbol) -> Unit) {
        useSiteMemberScope.processDeclaredConstructors process@{ original ->
            val function = signatureEnhancement.enhancedConstructor(original)
            processor(function)
        }
    }

    override fun processDirectOverriddenFunctionsWithBaseScope(
        functionSymbol: FirNamedFunctionSymbol,
        processor: (FirNamedFunctionSymbol, FirTypeScope) -> ProcessorAction
    ): ProcessorAction =
        doProcessDirectOverriddenCallables(
            functionSymbol, processor, enhancedToOriginalFunctions, FirTypeScope::processDirectOverriddenFunctionsWithBaseScope
        )

    override fun processDirectOverriddenPropertiesWithBaseScope(
        propertySymbol: FirPropertySymbol,
        processor: (FirPropertySymbol, FirTypeScope) -> ProcessorAction
    ): ProcessorAction = doProcessDirectOverriddenCallables(
        propertySymbol, processor, enhancedToOriginalProperties, FirTypeScope::processDirectOverriddenPropertiesWithBaseScope
    )

    private fun <S : FirCallableSymbol<*>> doProcessDirectOverriddenCallables(
        callableSymbol: S,
        processor: (S, FirTypeScope) -> ProcessorAction,
        enhancedToOriginalMap: Map<S, S>,
        processDirectOverriddenCallables: FirTypeScope.(S, (S, FirTypeScope) -> ProcessorAction) -> ProcessorAction
    ): ProcessorAction {
        val unwrappedSymbol = if (callableSymbol.origin == FirDeclarationOrigin.RenamedForOverride) {
            @Suppress("UNCHECKED_CAST")
            callableSymbol.fir.initialSignatureAttr as? S ?: callableSymbol
        } else {
            callableSymbol
        }
        val original = enhancedToOriginalMap[unwrappedSymbol] ?: return ProcessorAction.NONE
        return useSiteMemberScope.processDirectOverriddenCallables(original, processor)
    }

    override fun toString(): String {
        return "Java enhancement scope for ${owner.classId}"
    }

    @DelicateScopeAPI
    override fun withReplacedSessionOrNull(
        newSession: FirSession,
        newScopeSession: ScopeSession
    ): JavaClassMembersEnhancementScope {
        return JavaClassMembersEnhancementScope(
            newSession, owner,
            useSiteMemberScope.withReplacedSessionOrNull(newSession, newScopeSession)
        )
    }
}
