/*
 * Copyright 2010-2025 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.resolve

import ksp.org.jetbrains.kotlin.KtFakeSourceElementKind
import ksp.org.jetbrains.kotlin.fakeElement
import ksp.org.jetbrains.kotlin.fir.FirSession
import ksp.org.jetbrains.kotlin.fir.declarations.utils.isSealed
import ksp.org.jetbrains.kotlin.fir.diagnostics.ConeDiagnostic
import ksp.org.jetbrains.kotlin.fir.expressions.FirExpression
import ksp.org.jetbrains.kotlin.fir.expressions.FirPropertyAccessExpression
import ksp.org.jetbrains.kotlin.fir.expressions.FirResolvedQualifier
import ksp.org.jetbrains.kotlin.fir.expressions.builder.buildPropertyAccessExpression
import ksp.org.jetbrains.kotlin.fir.expressions.builder.buildResolvedQualifier
import ksp.org.jetbrains.kotlin.fir.references.FirErrorNamedReference
import ksp.org.jetbrains.kotlin.fir.references.FirResolvedErrorReference
import ksp.org.jetbrains.kotlin.fir.references.FirResolvedNamedReference
import ksp.org.jetbrains.kotlin.fir.references.builder.buildSimpleNamedReference
import ksp.org.jetbrains.kotlin.fir.resolve.calls.candidate.FirErrorReferenceWithCandidate
import ksp.org.jetbrains.kotlin.fir.resolve.diagnostics.ConeAmbiguityError
import ksp.org.jetbrains.kotlin.fir.resolve.diagnostics.ConeHiddenCandidateError
import ksp.org.jetbrains.kotlin.fir.resolve.diagnostics.ConeUnresolvedError
import ksp.org.jetbrains.kotlin.fir.resolve.diagnostics.ConeVisibilityError
import ksp.org.jetbrains.kotlin.fir.symbols.impl.*
import ksp.org.jetbrains.kotlin.fir.types.*
import ksp.org.jetbrains.kotlin.resolve.calls.tower.CandidateApplicability

fun ConeKotlinType.getClassRepresentativeForContextSensitiveResolution(session: FirSession): FirRegularClassSymbol? {
    return when (this) {
        is ConeFlexibleType ->
            lowerBound.getClassRepresentativeForContextSensitiveResolution(session)?.takeIf {
                it == upperBound.getClassRepresentativeForContextSensitiveResolution(session)
            }

        is ConeDefinitelyNotNullType -> original.getClassRepresentativeForContextSensitiveResolution(session)

        is ConeIntegerLiteralType -> possibleTypes.singleOrNull()?.getClassRepresentativeForContextSensitiveResolution(session)

        is ConeIntersectionType -> {
            val representativesForComponents =
                intersectedTypes.map { it.getClassRepresentativeForContextSensitiveResolution(session) }

            if (representativesForComponents.any { it == null }) return null
            @Suppress("UNCHECKED_CAST") // See the check above
            representativesForComponents as List<FirClassSymbol<*>>

            representativesForComponents.firstOrNull { candidate ->
                representativesForComponents.all { other ->
                    candidate.fir.isSubclassOf(other.toLookupTag(), session, isStrict = false)
                }
            }
        }

        is ConeLookupTagBasedType ->
            when (val symbol = lookupTag.toSymbol(session)) {
                is FirRegularClassSymbol -> symbol

                is FirTypeParameterSymbol ->
                    symbol.resolvedBounds.singleOrNull()?.coneType?.getClassRepresentativeForContextSensitiveResolution(session)

                is FirAnonymousObjectSymbol -> null
                is FirTypeAliasSymbol ->
                    fullyExpandedType(session)
                        .takeIf { it !== this }
                        ?.getClassRepresentativeForContextSensitiveResolution(session)
                null -> null
            }

        is ConeCapturedType, is ConeStubTypeForTypeVariableInSubtyping, is ConeTypeVariableType -> null
    }
}

fun FirRegularClassSymbol.getParentChainForContextSensitiveResolution(session: FirSession): Sequence<FirRegularClassSymbol> = sequence {
    var current: FirRegularClassSymbol? = this@getParentChainForContextSensitiveResolution

    while (current != null) {
        yield(current)
        current = (current.getContainingDeclaration(session) as? FirRegularClassSymbol)
            ?.takeIf { it.isSealed }
            ?.takeIf { isSubclassOf(it.toLookupTag(), session, isStrict = true, lookupInterfaces = true) }
    }
}

/**
 * @return not-nullable value when resolution was successful
 */
fun BodyResolveComponents.runContextSensitiveResolutionForPropertyAccess(
    originalExpression: FirPropertyAccessExpression,
    expectedType: ConeKotlinType,
): FirExpression? {
    val representativeClass: FirRegularClassSymbol =
        expectedType.getClassRepresentativeForContextSensitiveResolution(session)
            ?: return null

    for (representativeClass in representativeClass.getParentChainForContextSensitiveResolution(session)) {
        val additionalQualifier = representativeClass.toImplicitResolvedQualifierReceiver(
            this,
            originalExpression.source?.fakeElement(KtFakeSourceElementKind.QualifierForContextSensitiveResolution)
        )

        val newAccess = buildPropertyAccessExpression {
            explicitReceiver = additionalQualifier
            source = originalExpression.source
            calleeReference = buildSimpleNamedReference {
                source = originalExpression.calleeReference.source
                name = originalExpression.calleeReference.name
            }
        }

        val newExpression = callResolver.resolveVariableAccessAndSelectCandidate(
            newAccess,
            isUsedAsReceiver = false, isUsedAsGetClassReceiver = false,
            callSite = newAccess,
            ResolutionMode.ContextIndependent,
        )


        val shouldTakeNewExpression = when (newExpression) {
            is FirPropertyAccessExpression -> {
                val newCalleeReference = newExpression.calleeReference
                newCalleeReference is FirResolvedNamedReference && newCalleeReference !is FirResolvedErrorReference
            }

            // resolved qualifiers are always successful when returned
            is FirResolvedQualifier -> true

            // Non-trivial FIR element
            else -> false
        }

        if (shouldTakeNewExpression) return newExpression
    }

    return null
}

fun FirPropertyAccessExpression.shouldBeResolvedInContextSensitiveMode(): Boolean {
    val diagnostic = when (val calleeReference = calleeReference) {
        is FirErrorNamedReference -> calleeReference.diagnostic
        is FirErrorReferenceWithCandidate -> calleeReference.diagnostic
        is FirResolvedErrorReference -> calleeReference.diagnostic
        else -> return false
    }

    // Only simple name expressions are supported
    if (explicitReceiver != null) return false

    return diagnostic.meansNoAvailableCandidate()
}

private fun ConeDiagnostic.meansNoAvailableCandidate(): Boolean =
    when (this) {
        is ConeUnresolvedError, is ConeVisibilityError, is ConeHiddenCandidateError -> true
        is ConeAmbiguityError -> candidates.all {
            it.applicability == CandidateApplicability.HIDDEN || it.applicability == CandidateApplicability.K2_VISIBILITY_ERROR
        }
        else -> false
    }
