/*
 * Decompiled with CFR 0.152.
 */
package com.google.firebase.remoteconfig.internal;

import android.content.Context;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.os.Build;
import android.util.Log;
import androidx.annotation.Keep;
import androidx.annotation.VisibleForTesting;
import androidx.core.content.pm.PackageInfoCompat;
import com.google.android.gms.common.util.AndroidUtilsLight;
import com.google.android.gms.common.util.Hex;
import com.google.firebase.remoteconfig.FirebaseRemoteConfigClientException;
import com.google.firebase.remoteconfig.FirebaseRemoteConfigException;
import com.google.firebase.remoteconfig.FirebaseRemoteConfigServerException;
import com.google.firebase.remoteconfig.internal.ConfigContainer;
import com.google.firebase.remoteconfig.internal.ConfigFetchHandler;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLConnection;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.TimeZone;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

public class ConfigFetchHttpClient {
    private static final String API_KEY_HEADER = "X-Goog-Api-Key";
    private static final String ETAG_HEADER = "ETag";
    private static final String IF_NONE_MATCH_HEADER = "If-None-Match";
    private static final String X_ANDROID_PACKAGE_HEADER = "X-Android-Package";
    private static final String X_ANDROID_CERT_HEADER = "X-Android-Cert";
    private static final String X_GOOGLE_GFE_CAN_RETRY = "X-Google-GFE-Can-Retry";
    private static final String INSTALLATIONS_AUTH_TOKEN_HEADER = "X-Goog-Firebase-Installations-Auth";
    private final Context context;
    private final String appId;
    private final String apiKey;
    private final String projectNumber;
    private final String namespace;
    private final long connectTimeoutInSeconds;
    private final long readTimeoutInSeconds;
    private static final String ISO_DATE_PATTERN = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'";
    private static final Pattern GMP_APP_ID_PATTERN = Pattern.compile("^[^:]+:([0-9]+):(android|ios|web):([0-9a-f]+)");

    public ConfigFetchHttpClient(Context context, String appId, String apiKey, String namespace, long connectTimeoutInSeconds, long readTimeoutInSeconds) {
        this.context = context;
        this.appId = appId;
        this.apiKey = apiKey;
        this.projectNumber = ConfigFetchHttpClient.extractProjectNumberFromAppId(appId);
        this.namespace = namespace;
        this.connectTimeoutInSeconds = connectTimeoutInSeconds;
        this.readTimeoutInSeconds = readTimeoutInSeconds;
    }

    @VisibleForTesting
    public long getConnectTimeoutInSeconds() {
        return this.connectTimeoutInSeconds;
    }

    @VisibleForTesting
    public long getReadTimeoutInSeconds() {
        return this.readTimeoutInSeconds;
    }

    private static String extractProjectNumberFromAppId(String gmpAppId) {
        Matcher matcher = GMP_APP_ID_PATTERN.matcher(gmpAppId);
        return matcher.matches() ? matcher.group(1) : null;
    }

    HttpURLConnection createHttpURLConnection() throws FirebaseRemoteConfigException {
        try {
            URL url = new URL(this.getFetchUrl(this.projectNumber, this.namespace));
            return (HttpURLConnection)url.openConnection();
        }
        catch (IOException e) {
            throw new FirebaseRemoteConfigException(e.getMessage());
        }
    }

    @Keep
    ConfigFetchHandler.FetchResponse fetch(HttpURLConnection urlConnection, String installationId, String installationAuthToken, Map<String, String> analyticsUserProperties, String lastFetchETag, Map<String, String> customHeaders, Long firstOpenTime, Date currentTime) throws FirebaseRemoteConfigException {
        JSONObject fetchResponse;
        String fetchResponseETag;
        this.setUpUrlConnection(urlConnection, lastFetchETag, installationAuthToken, customHeaders);
        try {
            byte[] requestBody = this.createFetchRequestBody(installationId, installationAuthToken, analyticsUserProperties, firstOpenTime).toString().getBytes("utf-8");
            this.setFetchRequestBody(urlConnection, requestBody);
            urlConnection.connect();
            int responseCode = urlConnection.getResponseCode();
            if (responseCode != 200) {
                throw new FirebaseRemoteConfigServerException(responseCode, urlConnection.getResponseMessage());
            }
            fetchResponseETag = urlConnection.getHeaderField(ETAG_HEADER);
            fetchResponse = this.getFetchResponseBody(urlConnection);
        }
        catch (IOException | JSONException e) {
            throw new FirebaseRemoteConfigClientException("The client had an error while calling the backend!", e);
        }
        finally {
            urlConnection.disconnect();
            try {
                urlConnection.getInputStream().close();
            }
            catch (IOException iOException) {}
        }
        if (!this.backendHasUpdates(fetchResponse)) {
            return ConfigFetchHandler.FetchResponse.forBackendHasNoUpdates(currentTime);
        }
        ConfigContainer fetchedConfigs = ConfigFetchHttpClient.extractConfigs(fetchResponse, currentTime);
        return ConfigFetchHandler.FetchResponse.forBackendUpdatesFetched(fetchedConfigs, fetchResponseETag);
    }

    private void setUpUrlConnection(HttpURLConnection urlConnection, String lastFetchEtag, String installationAuthToken, Map<String, String> customHeaders) {
        urlConnection.setDoOutput(true);
        urlConnection.setConnectTimeout((int)TimeUnit.SECONDS.toMillis(this.connectTimeoutInSeconds));
        urlConnection.setReadTimeout((int)TimeUnit.SECONDS.toMillis(this.readTimeoutInSeconds));
        urlConnection.setRequestProperty(IF_NONE_MATCH_HEADER, lastFetchEtag);
        this.setCommonRequestHeaders(urlConnection, installationAuthToken);
        this.setCustomRequestHeaders(urlConnection, customHeaders);
    }

    private String getFetchUrl(String projectNumber, String namespace) {
        return String.format("https://firebaseremoteconfig.googleapis.com/v1/projects/%s/namespaces/%s:fetch", projectNumber, namespace);
    }

    private void setCommonRequestHeaders(HttpURLConnection urlConnection, String installationAuthToken) {
        urlConnection.setRequestProperty(API_KEY_HEADER, this.apiKey);
        urlConnection.setRequestProperty(X_ANDROID_PACKAGE_HEADER, this.context.getPackageName());
        urlConnection.setRequestProperty(X_ANDROID_CERT_HEADER, this.getFingerprintHashForPackage());
        urlConnection.setRequestProperty(X_GOOGLE_GFE_CAN_RETRY, "yes");
        urlConnection.setRequestProperty(INSTALLATIONS_AUTH_TOKEN_HEADER, installationAuthToken);
        urlConnection.setRequestProperty("Content-Type", "application/json");
        urlConnection.setRequestProperty("Accept", "application/json");
    }

    private void setCustomRequestHeaders(HttpURLConnection urlConnection, Map<String, String> customHeaders) {
        for (Map.Entry<String, String> customHeaderEntry : customHeaders.entrySet()) {
            urlConnection.setRequestProperty(customHeaderEntry.getKey(), customHeaderEntry.getValue());
        }
    }

    private String getFingerprintHashForPackage() {
        try {
            byte[] hash = AndroidUtilsLight.getPackageCertificateHashBytes((Context)this.context, (String)this.context.getPackageName());
            if (hash == null) {
                Log.e((String)"FirebaseRemoteConfig", (String)("Could not get fingerprint hash for package: " + this.context.getPackageName()));
                return null;
            }
            return Hex.bytesToStringUppercase((byte[])hash, (boolean)false);
        }
        catch (PackageManager.NameNotFoundException e) {
            Log.e((String)"FirebaseRemoteConfig", (String)("No such package: " + this.context.getPackageName()), (Throwable)e);
            return null;
        }
    }

    private JSONObject createFetchRequestBody(String installationId, String installationAuthToken, Map<String, String> analyticsUserProperties, Long firstOpenTime) throws FirebaseRemoteConfigClientException {
        HashMap<String, String> requestBodyMap = new HashMap<String, String>();
        if (installationId == null) {
            throw new FirebaseRemoteConfigClientException("Fetch failed: Firebase installation id is null.");
        }
        requestBodyMap.put("appInstanceId", installationId);
        requestBodyMap.put("appInstanceIdToken", installationAuthToken);
        requestBodyMap.put("appId", this.appId);
        Locale locale = this.context.getResources().getConfiguration().locale;
        requestBodyMap.put("countryCode", locale.getCountry());
        String languageCode = Build.VERSION.SDK_INT >= 21 ? locale.toLanguageTag() : locale.toString();
        requestBodyMap.put("languageCode", languageCode);
        requestBodyMap.put("platformVersion", Integer.toString(Build.VERSION.SDK_INT));
        requestBodyMap.put("timeZone", TimeZone.getDefault().getID());
        try {
            PackageInfo packageInfo = this.context.getPackageManager().getPackageInfo(this.context.getPackageName(), 0);
            if (packageInfo != null) {
                requestBodyMap.put("appVersion", packageInfo.versionName);
                requestBodyMap.put("appBuild", Long.toString(PackageInfoCompat.getLongVersionCode((PackageInfo)packageInfo)));
            }
        }
        catch (PackageManager.NameNotFoundException nameNotFoundException) {
            // empty catch block
        }
        requestBodyMap.put("packageName", this.context.getPackageName());
        requestBodyMap.put("sdkVersion", "21.2.1");
        requestBodyMap.put("analyticsUserProperties", (String)new JSONObject(analyticsUserProperties));
        if (firstOpenTime != null) {
            requestBodyMap.put("firstOpenTime", this.convertToISOString(firstOpenTime));
        }
        return new JSONObject(requestBodyMap);
    }

    private String convertToISOString(long millisFromEpoch) {
        SimpleDateFormat isoDateFormat = new SimpleDateFormat(ISO_DATE_PATTERN, Locale.US);
        isoDateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
        return isoDateFormat.format(millisFromEpoch);
    }

    private void setFetchRequestBody(HttpURLConnection urlConnection, byte[] requestBody) throws IOException {
        urlConnection.setFixedLengthStreamingMode(requestBody.length);
        BufferedOutputStream out = new BufferedOutputStream(urlConnection.getOutputStream());
        ((OutputStream)out).write(requestBody);
        ((OutputStream)out).flush();
        ((OutputStream)out).close();
    }

    private JSONObject getFetchResponseBody(URLConnection urlConnection) throws IOException, JSONException {
        BufferedReader br = new BufferedReader(new InputStreamReader(urlConnection.getInputStream(), "utf-8"));
        StringBuilder responseStringBuilder = new StringBuilder();
        int current = 0;
        while ((current = br.read()) != -1) {
            responseStringBuilder.append((char)current);
        }
        return new JSONObject(responseStringBuilder.toString());
    }

    private boolean backendHasUpdates(JSONObject response) {
        try {
            return !response.get("state").equals("NO_CHANGE");
        }
        catch (JSONException e) {
            return true;
        }
    }

    private static ConfigContainer extractConfigs(JSONObject fetchResponse, Date fetchTime) throws FirebaseRemoteConfigClientException {
        try {
            ConfigContainer.Builder containerBuilder = ConfigContainer.newBuilder().withFetchTime(fetchTime);
            JSONObject entries = null;
            try {
                entries = fetchResponse.getJSONObject("entries");
            }
            catch (JSONException jSONException) {
                // empty catch block
            }
            if (entries != null) {
                containerBuilder = containerBuilder.replaceConfigsWith(entries);
            }
            JSONArray experimentDescriptions = null;
            try {
                experimentDescriptions = fetchResponse.getJSONArray("experimentDescriptions");
            }
            catch (JSONException jSONException) {
                // empty catch block
            }
            if (experimentDescriptions != null) {
                containerBuilder = containerBuilder.withAbtExperiments(experimentDescriptions);
            }
            JSONObject personalizationMetadata = null;
            try {
                personalizationMetadata = fetchResponse.getJSONObject("personalizationMetadata");
            }
            catch (JSONException jSONException) {
                // empty catch block
            }
            if (personalizationMetadata != null) {
                containerBuilder = containerBuilder.withPersonalizationMetadata(personalizationMetadata);
            }
            return containerBuilder.build();
        }
        catch (JSONException e) {
            throw new FirebaseRemoteConfigClientException("Fetch failed: fetch response could not be parsed.", e);
        }
    }
}

