/*
 * Decompiled with CFR 0.152.
 */
package com.google.firebase.remoteconfig.internal;

import android.text.format.DateUtils;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import androidx.annotation.WorkerThread;
import com.google.android.gms.common.util.Clock;
import com.google.android.gms.tasks.Task;
import com.google.android.gms.tasks.Tasks;
import com.google.firebase.analytics.connector.AnalyticsConnector;
import com.google.firebase.inject.Provider;
import com.google.firebase.installations.FirebaseInstallationsApi;
import com.google.firebase.installations.InstallationTokenResult;
import com.google.firebase.remoteconfig.FirebaseRemoteConfigClientException;
import com.google.firebase.remoteconfig.FirebaseRemoteConfigException;
import com.google.firebase.remoteconfig.FirebaseRemoteConfigFetchThrottledException;
import com.google.firebase.remoteconfig.FirebaseRemoteConfigServerException;
import com.google.firebase.remoteconfig.internal.ConfigCacheClient;
import com.google.firebase.remoteconfig.internal.ConfigContainer;
import com.google.firebase.remoteconfig.internal.ConfigFetchHttpClient;
import com.google.firebase.remoteconfig.internal.ConfigSharedPrefsClient;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.net.HttpURLConnection;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Random;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;

public class ConfigFetchHandler {
    public static final long DEFAULT_MINIMUM_FETCH_INTERVAL_IN_SECONDS = TimeUnit.HOURS.toSeconds(12L);
    @VisibleForTesting
    static final int[] BACKOFF_TIME_DURATIONS_IN_MINUTES = new int[]{2, 4, 8, 16, 32, 64, 128, 256};
    @VisibleForTesting
    static final int HTTP_TOO_MANY_REQUESTS = 429;
    @VisibleForTesting
    static final String FIRST_OPEN_TIME_KEY = "_fot";
    private static final String X_FIREBASE_RC_FETCH_TYPE = "X-Firebase-RC-Fetch-Type";
    private final FirebaseInstallationsApi firebaseInstallations;
    private final Provider<AnalyticsConnector> analyticsConnector;
    private final Executor executor;
    private final Clock clock;
    private final Random randomGenerator;
    private final ConfigCacheClient fetchedConfigsCache;
    private final ConfigFetchHttpClient frcBackendApiClient;
    private final ConfigSharedPrefsClient frcSharedPrefs;
    private final Map<String, String> customHttpHeaders;

    public ConfigFetchHandler(FirebaseInstallationsApi firebaseInstallations, Provider<AnalyticsConnector> analyticsConnector, Executor executor, Clock clock, Random randomGenerator, ConfigCacheClient fetchedConfigsCache, ConfigFetchHttpClient frcBackendApiClient, ConfigSharedPrefsClient frcSharedPrefs, Map<String, String> customHttpHeaders) {
        this.firebaseInstallations = firebaseInstallations;
        this.analyticsConnector = analyticsConnector;
        this.executor = executor;
        this.clock = clock;
        this.randomGenerator = randomGenerator;
        this.fetchedConfigsCache = fetchedConfigsCache;
        this.frcBackendApiClient = frcBackendApiClient;
        this.frcSharedPrefs = frcSharedPrefs;
        this.customHttpHeaders = customHttpHeaders;
    }

    public Task<FetchResponse> fetch() {
        return this.fetch(this.frcSharedPrefs.getMinimumFetchIntervalInSeconds());
    }

    public Task<FetchResponse> fetch(long minimumFetchIntervalInSeconds) {
        HashMap<String, String> copyOfCustomHttpHeaders = new HashMap<String, String>(this.customHttpHeaders);
        copyOfCustomHttpHeaders.put(X_FIREBASE_RC_FETCH_TYPE, FetchType.BASE.getValue() + "/" + 1);
        return this.fetchedConfigsCache.get().continueWithTask(this.executor, cachedFetchConfigsTask -> this.fetchIfCacheExpiredAndNotThrottled((Task<ConfigContainer>)cachedFetchConfigsTask, minimumFetchIntervalInSeconds, copyOfCustomHttpHeaders));
    }

    public Task<FetchResponse> fetchNowWithTypeAndAttemptNumber(FetchType fetchType, int fetchAttemptNumber) {
        HashMap<String, String> copyOfCustomHttpHeaders = new HashMap<String, String>(this.customHttpHeaders);
        copyOfCustomHttpHeaders.put(X_FIREBASE_RC_FETCH_TYPE, fetchType.getValue() + "/" + fetchAttemptNumber);
        return this.fetchedConfigsCache.get().continueWithTask(this.executor, cachedFetchConfigsTask -> this.fetchIfCacheExpiredAndNotThrottled((Task<ConfigContainer>)cachedFetchConfigsTask, 0L, copyOfCustomHttpHeaders));
    }

    private Task<FetchResponse> fetchIfCacheExpiredAndNotThrottled(Task<ConfigContainer> cachedFetchConfigsTask, long minimumFetchIntervalInSeconds, Map<String, String> customFetchHeaders) {
        Task fetchResponseTask;
        Date currentTime = new Date(this.clock.currentTimeMillis());
        if (cachedFetchConfigsTask.isSuccessful() && this.areCachedFetchConfigsValid(minimumFetchIntervalInSeconds, currentTime)) {
            return Tasks.forResult((Object)FetchResponse.forLocalStorageUsed(currentTime));
        }
        Date backoffEndTime = this.getBackoffEndTimeInMillis(currentTime);
        if (backoffEndTime != null) {
            fetchResponseTask = Tasks.forException((Exception)((Object)new FirebaseRemoteConfigFetchThrottledException(this.createThrottledMessage(backoffEndTime.getTime() - currentTime.getTime()), backoffEndTime.getTime())));
        } else {
            Task installationIdTask = this.firebaseInstallations.getId();
            Task installationAuthTokenTask = this.firebaseInstallations.getToken(false);
            fetchResponseTask = Tasks.whenAllComplete((Task[])new Task[]{installationIdTask, installationAuthTokenTask}).continueWithTask(this.executor, completedInstallationsTasks -> {
                if (!installationIdTask.isSuccessful()) {
                    return Tasks.forException((Exception)((Object)new FirebaseRemoteConfigClientException("Firebase Installations failed to get installation ID for fetch.", installationIdTask.getException())));
                }
                if (!installationAuthTokenTask.isSuccessful()) {
                    return Tasks.forException((Exception)((Object)new FirebaseRemoteConfigClientException("Firebase Installations failed to get installation auth token for fetch.", installationAuthTokenTask.getException())));
                }
                String installationId = (String)installationIdTask.getResult();
                String installationToken = ((InstallationTokenResult)installationAuthTokenTask.getResult()).getToken();
                return this.fetchFromBackendAndCacheResponse(installationId, installationToken, currentTime, customFetchHeaders);
            });
        }
        return fetchResponseTask.continueWithTask(this.executor, completedFetchTask -> {
            this.updateLastFetchStatusAndTime((Task<FetchResponse>)completedFetchTask, currentTime);
            return completedFetchTask;
        });
    }

    private boolean areCachedFetchConfigsValid(long cacheExpirationInSeconds, Date newFetchTime) {
        Date lastSuccessfulFetchTime = this.frcSharedPrefs.getLastSuccessfulFetchTime();
        if (lastSuccessfulFetchTime.equals(ConfigSharedPrefsClient.LAST_FETCH_TIME_NO_FETCH_YET)) {
            return false;
        }
        Date cacheExpirationTime = new Date(lastSuccessfulFetchTime.getTime() + TimeUnit.SECONDS.toMillis(cacheExpirationInSeconds));
        return newFetchTime.before(cacheExpirationTime);
    }

    @Nullable
    private Date getBackoffEndTimeInMillis(Date currentTime) {
        Date backoffEndTime = this.frcSharedPrefs.getBackoffMetadata().getBackoffEndTime();
        if (currentTime.before(backoffEndTime)) {
            return backoffEndTime;
        }
        return null;
    }

    private String createThrottledMessage(long throttledDurationInMillis) {
        return String.format("Fetch is throttled. Please wait before calling fetch again: %s", DateUtils.formatElapsedTime((long)TimeUnit.MILLISECONDS.toSeconds(throttledDurationInMillis)));
    }

    private Task<FetchResponse> fetchFromBackendAndCacheResponse(String installationId, String installationToken, Date fetchTime, Map<String, String> customFetchHeaders) {
        try {
            FetchResponse fetchResponse = this.fetchFromBackend(installationId, installationToken, fetchTime, customFetchHeaders);
            if (fetchResponse.getStatus() != 0) {
                return Tasks.forResult((Object)fetchResponse);
            }
            return this.fetchedConfigsCache.put(fetchResponse.getFetchedConfigs()).onSuccessTask(this.executor, putContainer -> Tasks.forResult((Object)fetchResponse));
        }
        catch (FirebaseRemoteConfigException frce) {
            return Tasks.forException((Exception)((Object)frce));
        }
    }

    @WorkerThread
    private FetchResponse fetchFromBackend(String installationId, String installationToken, Date currentTime, Map<String, String> customFetchHeaders) throws FirebaseRemoteConfigException {
        try {
            HttpURLConnection urlConnection = this.frcBackendApiClient.createHttpURLConnection();
            FetchResponse response = this.frcBackendApiClient.fetch(urlConnection, installationId, installationToken, this.getUserProperties(), this.frcSharedPrefs.getLastFetchETag(), customFetchHeaders, this.getFirstOpenTime(), currentTime, this.frcSharedPrefs.getCustomSignals());
            if (response.getFetchedConfigs() != null) {
                this.frcSharedPrefs.setLastTemplateVersion(response.getFetchedConfigs().getTemplateVersionNumber());
            }
            if (response.getLastFetchETag() != null) {
                this.frcSharedPrefs.setLastFetchETag(response.getLastFetchETag());
            }
            this.frcSharedPrefs.resetBackoff();
            return response;
        }
        catch (FirebaseRemoteConfigServerException serverHttpError) {
            ConfigSharedPrefsClient.BackoffMetadata backoffMetadata = this.updateAndReturnBackoffMetadata(serverHttpError.getHttpStatusCode(), currentTime);
            if (this.shouldThrottle(backoffMetadata, serverHttpError.getHttpStatusCode())) {
                throw new FirebaseRemoteConfigFetchThrottledException(backoffMetadata.getBackoffEndTime().getTime());
            }
            throw this.createExceptionWithGenericMessage(serverHttpError);
        }
    }

    private FirebaseRemoteConfigServerException createExceptionWithGenericMessage(FirebaseRemoteConfigServerException httpError) throws FirebaseRemoteConfigClientException {
        String errorMessage;
        switch (httpError.getHttpStatusCode()) {
            case 401: {
                errorMessage = "The request did not have the required credentials. Please make sure your google-services.json is valid.";
                break;
            }
            case 403: {
                errorMessage = "The user is not authorized to access the project. Please make sure you are using the API key that corresponds to your Firebase project.";
                break;
            }
            case 500: {
                errorMessage = "There was an internal server error.";
                break;
            }
            case 502: 
            case 503: 
            case 504: {
                errorMessage = "The server is unavailable. Please try again later.";
                break;
            }
            case 429: {
                throw new FirebaseRemoteConfigClientException("The throttled response from the server was not handled correctly by the FRC SDK.");
            }
            default: {
                errorMessage = "The server returned an unexpected error.";
            }
        }
        return new FirebaseRemoteConfigServerException(httpError.getHttpStatusCode(), "Fetch failed: " + errorMessage, (Throwable)((Object)httpError));
    }

    private ConfigSharedPrefsClient.BackoffMetadata updateAndReturnBackoffMetadata(int statusCode, Date currentTime) {
        if (this.isThrottleableServerError(statusCode)) {
            this.updateBackoffMetadataWithLastFailedFetchTime(currentTime);
        }
        return this.frcSharedPrefs.getBackoffMetadata();
    }

    private boolean isThrottleableServerError(int httpStatusCode) {
        return httpStatusCode == 429 || httpStatusCode == 502 || httpStatusCode == 503 || httpStatusCode == 504;
    }

    private void updateBackoffMetadataWithLastFailedFetchTime(Date lastFailedFetchTime) {
        int numFailedFetches = this.frcSharedPrefs.getBackoffMetadata().getNumFailedFetches();
        long backoffDurationInMillis = this.getRandomizedBackoffDurationInMillis(++numFailedFetches);
        Date backoffEndTime = new Date(lastFailedFetchTime.getTime() + backoffDurationInMillis);
        this.frcSharedPrefs.setBackoffMetadata(numFailedFetches, backoffEndTime);
    }

    private long getRandomizedBackoffDurationInMillis(int numFailedFetches) {
        long timeOutDurationInMillis = TimeUnit.MINUTES.toMillis(BACKOFF_TIME_DURATIONS_IN_MINUTES[Math.min(numFailedFetches, BACKOFF_TIME_DURATIONS_IN_MINUTES.length) - 1]);
        return timeOutDurationInMillis / 2L + (long)this.randomGenerator.nextInt((int)timeOutDurationInMillis);
    }

    private boolean shouldThrottle(ConfigSharedPrefsClient.BackoffMetadata backoffMetadata, int httpStatusCode) {
        return backoffMetadata.getNumFailedFetches() > 1 || httpStatusCode == 429;
    }

    private void updateLastFetchStatusAndTime(Task<FetchResponse> completedFetchTask, Date fetchTime) {
        if (completedFetchTask.isSuccessful()) {
            this.frcSharedPrefs.updateLastFetchAsSuccessfulAt(fetchTime);
            return;
        }
        Exception fetchException = completedFetchTask.getException();
        if (fetchException == null) {
            return;
        }
        if (fetchException instanceof FirebaseRemoteConfigFetchThrottledException) {
            this.frcSharedPrefs.updateLastFetchAsThrottled();
        } else {
            this.frcSharedPrefs.updateLastFetchAsFailed();
        }
    }

    @WorkerThread
    private Map<String, String> getUserProperties() {
        HashMap<String, String> userPropertiesMap = new HashMap<String, String>();
        AnalyticsConnector connector = (AnalyticsConnector)this.analyticsConnector.get();
        if (connector == null) {
            return userPropertiesMap;
        }
        for (Map.Entry userPropertyEntry : connector.getUserProperties(false).entrySet()) {
            userPropertiesMap.put((String)userPropertyEntry.getKey(), userPropertyEntry.getValue().toString());
        }
        return userPropertiesMap;
    }

    @WorkerThread
    private Long getFirstOpenTime() {
        AnalyticsConnector connector = (AnalyticsConnector)this.analyticsConnector.get();
        if (connector == null) {
            return null;
        }
        return (Long)connector.getUserProperties(true).get(FIRST_OPEN_TIME_KEY);
    }

    public long getTemplateVersionNumber() {
        return this.frcSharedPrefs.getLastTemplateVersion();
    }

    @VisibleForTesting
    public Provider<AnalyticsConnector> getAnalyticsConnector() {
        return this.analyticsConnector;
    }

    public static enum FetchType {
        BASE("BASE"),
        REALTIME("REALTIME");

        private final String value;

        private FetchType(String value) {
            this.value = value;
        }

        String getValue() {
            return this.value;
        }
    }

    public static class FetchResponse {
        private final Date fetchTime;
        private final int status;
        private final ConfigContainer fetchedConfigs;
        @Nullable
        private final String lastFetchETag;

        private FetchResponse(Date fetchTime, int status, ConfigContainer fetchedConfigs, @Nullable String lastFetchETag) {
            this.fetchTime = fetchTime;
            this.status = status;
            this.fetchedConfigs = fetchedConfigs;
            this.lastFetchETag = lastFetchETag;
        }

        public static FetchResponse forBackendUpdatesFetched(ConfigContainer fetchedConfigs, String lastFetchETag) {
            return new FetchResponse(fetchedConfigs.getFetchTime(), 0, fetchedConfigs, lastFetchETag);
        }

        public static FetchResponse forBackendHasNoUpdates(Date fetchTime, ConfigContainer fetchedConfigs) {
            return new FetchResponse(fetchTime, 1, fetchedConfigs, null);
        }

        public static FetchResponse forLocalStorageUsed(Date fetchTime) {
            return new FetchResponse(fetchTime, 2, null, null);
        }

        Date getFetchTime() {
            return this.fetchTime;
        }

        @Nullable
        String getLastFetchETag() {
            return this.lastFetchETag;
        }

        int getStatus() {
            return this.status;
        }

        public ConfigContainer getFetchedConfigs() {
            return this.fetchedConfigs;
        }

        @Retention(value=RetentionPolicy.SOURCE)
        public static @interface Status {
            public static final int BACKEND_UPDATES_FETCHED = 0;
            public static final int BACKEND_HAS_NO_UPDATES = 1;
            public static final int LOCAL_STORAGE_USED = 2;
        }
    }
}

