/*
 * Decompiled with CFR 0.152.
 */
package com.google.firebase.firestore.core;

import android.annotation.SuppressLint;
import android.content.Context;
import androidx.annotation.Nullable;
import com.google.android.gms.tasks.Task;
import com.google.android.gms.tasks.TaskCompletionSource;
import com.google.android.gms.tasks.Tasks;
import com.google.firebase.firestore.AggregateField;
import com.google.firebase.firestore.EventListener;
import com.google.firebase.firestore.FirebaseFirestoreException;
import com.google.firebase.firestore.FirebaseFirestoreSettings;
import com.google.firebase.firestore.LoadBundleTask;
import com.google.firebase.firestore.TransactionOptions;
import com.google.firebase.firestore.auth.CredentialsProvider;
import com.google.firebase.firestore.auth.User;
import com.google.firebase.firestore.bundle.BundleReader;
import com.google.firebase.firestore.bundle.BundleSerializer;
import com.google.firebase.firestore.bundle.NamedQuery;
import com.google.firebase.firestore.core.ComponentProvider;
import com.google.firebase.firestore.core.DatabaseInfo;
import com.google.firebase.firestore.core.EventManager;
import com.google.firebase.firestore.core.MemoryComponentProvider;
import com.google.firebase.firestore.core.Query;
import com.google.firebase.firestore.core.QueryListener;
import com.google.firebase.firestore.core.SQLiteComponentProvider;
import com.google.firebase.firestore.core.SyncEngine;
import com.google.firebase.firestore.core.Target;
import com.google.firebase.firestore.core.Transaction;
import com.google.firebase.firestore.core.View;
import com.google.firebase.firestore.core.ViewSnapshot;
import com.google.firebase.firestore.local.IndexBackfiller;
import com.google.firebase.firestore.local.LocalStore;
import com.google.firebase.firestore.local.Persistence;
import com.google.firebase.firestore.local.QueryResult;
import com.google.firebase.firestore.local.Scheduler;
import com.google.firebase.firestore.model.Document;
import com.google.firebase.firestore.model.DocumentKey;
import com.google.firebase.firestore.model.FieldIndex;
import com.google.firebase.firestore.model.mutation.Mutation;
import com.google.firebase.firestore.remote.Datastore;
import com.google.firebase.firestore.remote.GrpcMetadataProvider;
import com.google.firebase.firestore.remote.RemoteSerializer;
import com.google.firebase.firestore.remote.RemoteStore;
import com.google.firebase.firestore.util.Assert;
import com.google.firebase.firestore.util.AsyncQueue;
import com.google.firebase.firestore.util.Function;
import com.google.firebase.firestore.util.Logger;
import com.google.firestore.v1.Value;
import java.io.InputStream;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.atomic.AtomicBoolean;

public final class FirestoreClient {
    private static final String LOG_TAG = "FirestoreClient";
    private static final int MAX_CONCURRENT_LIMBO_RESOLUTIONS = 100;
    private final DatabaseInfo databaseInfo;
    private final CredentialsProvider<User> authProvider;
    private final CredentialsProvider<String> appCheckProvider;
    private final AsyncQueue asyncQueue;
    private final BundleSerializer bundleSerializer;
    private final GrpcMetadataProvider metadataProvider;
    private Persistence persistence;
    private LocalStore localStore;
    private RemoteStore remoteStore;
    private SyncEngine syncEngine;
    private EventManager eventManager;
    @Nullable
    private Scheduler indexBackfillScheduler;
    @Nullable
    private Scheduler gcScheduler;

    public FirestoreClient(Context context, DatabaseInfo databaseInfo, FirebaseFirestoreSettings settings, CredentialsProvider<User> authProvider, CredentialsProvider<String> appCheckProvider, AsyncQueue asyncQueue, @Nullable GrpcMetadataProvider metadataProvider) {
        this.databaseInfo = databaseInfo;
        this.authProvider = authProvider;
        this.appCheckProvider = appCheckProvider;
        this.asyncQueue = asyncQueue;
        this.metadataProvider = metadataProvider;
        this.bundleSerializer = new BundleSerializer(new RemoteSerializer(databaseInfo.getDatabaseId()));
        TaskCompletionSource firstUser = new TaskCompletionSource();
        AtomicBoolean initialized = new AtomicBoolean(false);
        asyncQueue.enqueueAndForget(() -> {
            try {
                User initialUser = (User)Tasks.await((Task)firstUser.getTask());
                this.initialize(context, initialUser, settings);
            }
            catch (InterruptedException | ExecutionException e) {
                throw new RuntimeException(e);
            }
        });
        authProvider.setChangeListener(user -> {
            if (initialized.compareAndSet(false, true)) {
                Assert.hardAssert(!firstUser.getTask().isComplete(), "Already fulfilled first user task", new Object[0]);
                firstUser.setResult(user);
            } else {
                asyncQueue.enqueueAndForget(() -> {
                    Assert.hardAssert(this.syncEngine != null, "SyncEngine not yet initialized", new Object[0]);
                    Logger.debug(LOG_TAG, "Credential changed. Current user: %s", user.getUid());
                    this.syncEngine.handleCredentialChange((User)user);
                });
            }
        });
        appCheckProvider.setChangeListener(appCheckToken -> {});
    }

    public Task<Void> disableNetwork() {
        this.verifyNotTerminated();
        return this.asyncQueue.enqueue(() -> this.remoteStore.disableNetwork());
    }

    public Task<Void> enableNetwork() {
        this.verifyNotTerminated();
        return this.asyncQueue.enqueue(() -> this.remoteStore.enableNetwork());
    }

    public Task<Void> terminate() {
        this.authProvider.removeChangeListener();
        this.appCheckProvider.removeChangeListener();
        return this.asyncQueue.enqueueAndInitiateShutdown(() -> {
            this.remoteStore.shutdown();
            this.persistence.shutdown();
            if (this.gcScheduler != null) {
                this.gcScheduler.stop();
            }
            if (this.indexBackfillScheduler != null) {
                this.indexBackfillScheduler.stop();
            }
        });
    }

    public boolean isTerminated() {
        return this.asyncQueue.isShuttingDown();
    }

    public QueryListener listen(Query query, EventManager.ListenOptions options, EventListener<ViewSnapshot> listener) {
        this.verifyNotTerminated();
        QueryListener queryListener = new QueryListener(query, options, listener);
        this.asyncQueue.enqueueAndForget(() -> this.eventManager.addQueryListener(queryListener));
        return queryListener;
    }

    public void stopListening(QueryListener listener) {
        if (this.isTerminated()) {
            return;
        }
        this.asyncQueue.enqueueAndForget(() -> this.eventManager.removeQueryListener(listener));
    }

    @SuppressLint(value={"TaskMainThread"})
    public Task<Document> getDocumentFromLocalCache(DocumentKey docKey) {
        this.verifyNotTerminated();
        return this.asyncQueue.enqueue(() -> this.localStore.readDocument(docKey)).continueWith(result -> {
            Document document = (Document)result.getResult();
            if (document.isFoundDocument()) {
                return document;
            }
            if (document.isNoDocument()) {
                return null;
            }
            throw new FirebaseFirestoreException("Failed to get document from cache. (However, this document may exist on the server. Run again without setting source to CACHE to attempt to retrieve the document from the server.)", FirebaseFirestoreException.Code.UNAVAILABLE);
        });
    }

    public Task<ViewSnapshot> getDocumentsFromLocalCache(Query query) {
        this.verifyNotTerminated();
        return this.asyncQueue.enqueue(() -> {
            QueryResult queryResult = this.localStore.executeQuery(query, true);
            View view = new View(query, queryResult.getRemoteKeys());
            View.DocumentChanges viewDocChanges = view.computeDocChanges(queryResult.getDocuments());
            return view.applyChanges(viewDocChanges).getSnapshot();
        });
    }

    public Task<Void> write(List<Mutation> mutations) {
        this.verifyNotTerminated();
        TaskCompletionSource source = new TaskCompletionSource();
        this.asyncQueue.enqueueAndForget(() -> this.syncEngine.writeMutations(mutations, (TaskCompletionSource<Void>)source));
        return source.getTask();
    }

    public <TResult> Task<TResult> transaction(TransactionOptions options, Function<Transaction, Task<TResult>> updateFunction) {
        this.verifyNotTerminated();
        return AsyncQueue.callTask(this.asyncQueue.getExecutor(), () -> this.syncEngine.transaction(this.asyncQueue, options, updateFunction));
    }

    @SuppressLint(value={"TaskMainThread"})
    public Task<Map<String, Value>> runAggregateQuery(Query query, List<AggregateField> aggregateFields) {
        this.verifyNotTerminated();
        TaskCompletionSource result = new TaskCompletionSource();
        this.asyncQueue.enqueueAndForget(() -> this.syncEngine.runAggregateQuery(query, aggregateFields).addOnSuccessListener(data -> result.setResult(data)).addOnFailureListener(e -> result.setException(e)));
        return result.getTask();
    }

    public Task<Void> waitForPendingWrites() {
        this.verifyNotTerminated();
        TaskCompletionSource source = new TaskCompletionSource();
        this.asyncQueue.enqueueAndForget(() -> this.syncEngine.registerPendingWritesTask((TaskCompletionSource<Void>)source));
        return source.getTask();
    }

    private void initialize(Context context, User user, FirebaseFirestoreSettings settings) {
        Logger.debug(LOG_TAG, "Initializing. user=%s", user.getUid());
        Datastore datastore = new Datastore(this.databaseInfo, this.asyncQueue, this.authProvider, this.appCheckProvider, context, this.metadataProvider);
        ComponentProvider.Configuration configuration = new ComponentProvider.Configuration(context, this.asyncQueue, this.databaseInfo, datastore, user, 100, settings);
        MemoryComponentProvider provider = settings.isPersistenceEnabled() ? new SQLiteComponentProvider() : new MemoryComponentProvider();
        provider.initialize(configuration);
        this.persistence = provider.getPersistence();
        this.gcScheduler = provider.getGarbageCollectionScheduler();
        this.localStore = provider.getLocalStore();
        this.remoteStore = provider.getRemoteStore();
        this.syncEngine = provider.getSyncEngine();
        this.eventManager = provider.getEventManager();
        IndexBackfiller indexBackfiller = provider.getIndexBackfiller();
        if (this.gcScheduler != null) {
            this.gcScheduler.start();
        }
        if (indexBackfiller != null) {
            this.indexBackfillScheduler = indexBackfiller.getScheduler();
            this.indexBackfillScheduler.start();
        }
    }

    public void addSnapshotsInSyncListener(EventListener<Void> listener) {
        this.verifyNotTerminated();
        this.asyncQueue.enqueueAndForget(() -> this.eventManager.addSnapshotsInSyncListener(listener));
    }

    public void loadBundle(InputStream bundleData, LoadBundleTask resultTask) {
        this.verifyNotTerminated();
        BundleReader bundleReader = new BundleReader(this.bundleSerializer, bundleData);
        this.asyncQueue.enqueueAndForget(() -> this.syncEngine.loadBundle(bundleReader, resultTask));
    }

    public Task<Query> getNamedQuery(String queryName) {
        this.verifyNotTerminated();
        TaskCompletionSource completionSource = new TaskCompletionSource();
        this.asyncQueue.enqueueAndForget(() -> {
            NamedQuery namedQuery = this.localStore.getNamedQuery(queryName);
            if (namedQuery != null) {
                Target target = namedQuery.getBundledQuery().getTarget();
                completionSource.setResult((Object)new Query(target.getPath(), target.getCollectionGroup(), target.getFilters(), target.getOrderBy(), target.getLimit(), namedQuery.getBundledQuery().getLimitType(), target.getStartAt(), target.getEndAt()));
            } else {
                completionSource.setResult(null);
            }
        });
        return completionSource.getTask();
    }

    public Task<Void> configureFieldIndexes(List<FieldIndex> fieldIndices) {
        this.verifyNotTerminated();
        return this.asyncQueue.enqueue(() -> this.localStore.configureFieldIndexes(fieldIndices));
    }

    public void removeSnapshotsInSyncListener(EventListener<Void> listener) {
        if (this.isTerminated()) {
            return;
        }
        this.asyncQueue.enqueueAndForget(() -> this.eventManager.removeSnapshotsInSyncListener(listener));
    }

    private void verifyNotTerminated() {
        if (this.isTerminated()) {
            throw new IllegalStateException("The client has already been terminated");
        }
    }
}

