/*
 * Copyright (C) 2015 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.google.gms.googleservices

import java.util.HashSet
import java.util.SortedSet
import java.util.TreeSet
import java.util.regex.Matcher
import java.util.regex.Pattern
import org.gradle.BuildListener
import org.gradle.BuildResult
import org.gradle.api.artifacts.Configuration
import org.gradle.api.artifacts.Dependency
import org.gradle.api.artifacts.ProjectDependency
import org.gradle.api.initialization.Settings
import org.gradle.api.internal.artifacts.dependencies.DefaultProjectDependency
import org.gradle.api.invocation.Gradle
import org.gradle.api.GradleException
import org.gradle.api.Plugin
import org.gradle.api.Project

class GoogleServicesPlugin implements Plugin<Project> {

  public final static String JSON_FILE_NAME = 'google-services.json'

  public final static String MODULE_GROUP = "com.google.android.gms"
  public final static String MODULE_GROUP_FIREBASE = "com.google.firebase"
  public final static String MODULE_CORE = "firebase-core"
  public final static String MODULE_VERSION = "11.4.2"
  public final static String MINIMUM_VERSION = "9.0.0"
  public final static String GRANULAR_BUILD_VERSION = "14.0.0"
  // Some example of things that match this pattern are:
  // "aBunchOfFlavors/release"
  // "flavor/debug"
  // "test"
  // And here is an example with the capture groups in [square brackets]
  // [a][BunchOfFlavors]/[release]
  public final static Pattern VARIANT_PATTERN = ~/(?:([^\p{javaUpperCase}]+)((?:\p{javaUpperCase}[^\p{javaUpperCase}]*)*)\/)?([^\/]*)/
  // Some example of things that match this pattern are:
  // "TestTheFlavor"
  // "FlavorsOfTheRainbow"
  // "Test"
  // And here is an example with the capture groups in [square brackets]
  // "[Flavors][Of][The][Rainbow]"
  // Note: Pattern must be applied in a loop, not just once.
  public final static Pattern FLAVOR_PATTERN = ~/(\p{javaUpperCase}[^\p{javaUpperCase}]*)/
  // These are the plugin types and the set of associated plugins whose presence should be checked for.
  private final static enum PluginType{
    APPLICATION([
      "android",
      "com.android.application"
    ]),
    LIBRARY([
      "android-library",
      "com.android.library"
    ]),
    FEATURE([
      "android-feature",
      "com.android.feature"
    ]),
    MODEL_APPLICATION([
      "com.android.model.application"
    ]),
    MODEL_LIBRARY(["com.android.model.library"])
    public PluginType(Collection plugins) {
      this.plugins = plugins
    }
    private final Collection plugins
    public Collection plugins() {
      return plugins
    }
  }

  public static boolean isGranular = false
  // This will only be filled if isGranular remains false.
  public static Version targetVersion = new Version(MODULE_VERSION, MODULE_VERSION)

  @Override
  void apply(Project project) {
    for (PluginType pluginType : PluginType.values()) {
      for (String plugin : pluginType.plugins()) {
        if (project.plugins.hasPlugin(plugin)) {
          addDependency(project)
          setupPlugin(project, pluginType)
          return;
        }
      }
    }
    // If the google-service plugin is applied before any android plugin.
    // We should warn that google service plugin should be applied at
    // the bottom of build file.
    showWarningForPluginLocation(project)

    // Setup google-services plugin after android plugin is applied.
    project.plugins.withId("android", {
      setupPlugin(project, PluginType.APPLICATION)
    })
    project.plugins.withId("android-library", {
      setupPlugin(project, PluginType.LIBRARY)
    })
    project.plugins.withId("android-feature", {
      setupPlugin(project, PluginType.FEATURE)
    })

    // Add dependencies after the build file is evaluate and hopefully it
    // can be execute before android plugin process the dependencies.
    project.afterEvaluate({ addDependency(project) })
  }

  private void showWarningForPluginLocation(Project project) {
    project.getLogger().warn(
        "please apply google-services plugin at the bottom of the build file.")
  }

  private void addDependency(Project project) {
    def firebaseDependencies = findTargetVersion(project)

    // For granular builds we do not add dependencies, instead we set this flag so that
    // GoogleServicesTask can check make the appropriate checks.
    if (isGranular(firebaseDependencies)) {
      isGranular = true
      return
    }

    targetVersion = firebaseDependencies.values().find()

    // If the target version is not lower than the minimum version.
    if (versionCompare(targetVersion.getTrimmedVersion(), MINIMUM_VERSION) >= 0) {
      // A new configuration has been introduced in android plugin 3.0, if its available, use it.
      String dependencyType = project.getConfigurations().findByName("implementation") ? "implementation" : "compile"
      project.dependencies.add(dependencyType, MODULE_GROUP_FIREBASE + ':' + MODULE_CORE + ':' + targetVersion.getRawVersion())
    } else {
      throw new GradleException("Version: " + targetVersion.getRawVersion() + " is lower than the minimum version (" +
      MINIMUM_VERSION + ") required for google-services plugin.")
    }
  }

  private Map<String, Version> findTargetVersion(Project project) {
    def output = new HashMap<String>();
    def dependencies = getAllDependencies(project)
    for (def dependency : dependencies) {
      if (dependency == null) continue
        if (MODULE_GROUP.equalsIgnoreCase(dependency.getGroup()) ||
        MODULE_GROUP_FIREBASE.equalsIgnoreCase(dependency.getGroup())) {
          // Use the first version found in the dependencies.
          output.put(dependency.getName(), new Version(dependency.getVersion(), dependency.getVersion().split("-")[0]))
        }
    }
    if (output.isEmpty()) {
      // If none of the version for Google play services is found, default
      // version is used and a warning that google-services plugin should be
      // applied at the bottom of the build file.
      project.getLogger().warn(
          "google-services plugin could not detect any version for " +
          MODULE_GROUP + " or " + MODULE_GROUP_FIREBASE +
          ", default version: " + MODULE_VERSION + " will be used.")
      showWarningForPluginLocation(project)
      // If no version is found, use the default one for the plugin.
      output.put(MODULE_CORE, new Version(MODULE_VERSION, MODULE_VERSION))
    }
    return output;
  }


  private void setupPlugin(Project project, PluginType pluginType) {
    switch (pluginType) {
      case PluginType.APPLICATION:
        project.android.applicationVariants.all { variant ->
          handleVariant(project, variant)
        }
        break;
      case PluginType.LIBRARY:
        project.android.libraryVariants.all { variant ->
          handleVariant(project, variant)
        }
        break;
      case PluginType.FEATURE:
        project.android.featureVariants.all { variant ->
          handleVariant(project, variant)
        }
        break;
      case PluginType.MODEL_APPLICATION:
        project.model.android.applicationVariants.all { variant ->
          handleVariant(project, variant)
        }
        break;
      case PluginType.MODEL_LIBRARY:
        project.model.android.libraryVariants.all { variant ->
          handleVariant(project, variant)
        }
        break;
    }
  }


  private static void handleVariant(Project project,
      def variant) {

    File quickstartFile = null
    List<String> fileLocations = getJsonLocations("$variant.dirName")
    String searchedLocation = System.lineSeparator()
    for (String location : fileLocations) {
      File jsonFile = project.file(location + '/' + JSON_FILE_NAME)
      searchedLocation = searchedLocation + jsonFile.getPath() + System.lineSeparator()
      if (jsonFile.isFile()) {
        quickstartFile = jsonFile
        break
      }
    }

    if (quickstartFile == null) {
      project.getLogger().warn("Could not find $JSON_FILE_NAME while looking in $fileLocations");
      quickstartFile = project.file(JSON_FILE_NAME)
      searchedLocation = searchedLocation + quickstartFile.getPath()
    }

    File outputDir =
        project.file("$project.buildDir/generated/res/google-services/$variant.dirName")

    GoogleServicesTask task = project.tasks
        .create("process${variant.name.capitalize()}GoogleServices",
        GoogleServicesTask)

    task.quickstartFile = quickstartFile
    task.intermediateDir = outputDir
    task.packageName = variant.applicationId
    task.moduleGroup = MODULE_GROUP
    task.moduleGroupFirebase = MODULE_GROUP_FIREBASE
    task.isGranular = isGranular
    // Use the target version for the task.
    task.moduleVersion = targetVersion.getRawVersion();
    variant.registerResGeneratingTask(task, outputDir)
    task.searchedLocation = searchedLocation
  }

  private static List<String> splitVariantNames(String variant) {
    if (variant == null) {
      return Collections.emptyList();
    }
    List<String> flavors = new ArrayList<>()
    Matcher flavorMatcher = FLAVOR_PATTERN.matcher(variant)
    while (flavorMatcher.find()) {
      flavors.add(flavorMatcher.group(1).toLowerCase())
    }
    return flavors
  }

  private static long countSlashes(String input) {
    return input.codePoints().filter{x -> x == '/'}.count()
  }

  private static Set<Dependency> getAllDependencies(Project project) {
    def deps = new HashSet<Dependency>()
    getAllDependencies(project, deps);
    return deps
  }

  private static void getAllDependencies(Project project, Set<Dependency> deps) {
    def projectDependencies = project.getConfigurations()*.getAllDependencies().flatten()
    for (Dependency dep : projectDependencies) {
      if (!deps.contains(dep)) {
        if (dep instanceof ProjectDependency || dep instanceof DefaultProjectDependency) {
          def dependencyProject = dep.getDependencyProject()
          deps.add(dep)
          getAllDependencies(dependencyProject, deps)
        } else {
          deps.add(dep)
        }
      }
    }
  }

  static List<String> getJsonLocations(String variantDirname) {
    Matcher variantMatcher = VARIANT_PATTERN.matcher(variantDirname)
    variantMatcher.matches();
    List<String> flavorNames = new ArrayList<>()
    if (variantMatcher.group(1) != null) {
      flavorNames.add(variantMatcher.group(1).toLowerCase())
    }
    flavorNames.addAll(splitVariantNames(variantMatcher.group(2)))
    String buildType = variantMatcher.group(3)
    List<String> fileLocations = new ArrayList<>()
    String flavorName = "${variantMatcher.group(1)}${variantMatcher.group(2)}";
    fileLocations.add("src/$flavorName/$buildType")
    fileLocations.add("src/$buildType/$flavorName")
    fileLocations.add("src/$flavorName")
    fileLocations.add("src/$buildType")
    fileLocations.add("src/$flavorName${buildType.capitalize()}")
    fileLocations.add("src/$buildType")
    String fileLocation = "src"
    for(String flavor : flavorNames) {
      fileLocation += "/$flavor"
      fileLocations.add(fileLocation)
      fileLocations.add("$fileLocation/$buildType")
      fileLocations.add("$fileLocation${buildType.capitalize()}")
    }
    fileLocations.unique().sort{a,b -> countSlashes(b) <=> countSlashes(a)}
    return fileLocations
  }

  static int versionCompare(String str1, String str2) {
    String[] vals1 = str1.split("\\.");
    String[] vals2 = str2.split("\\.");
    int i = 0;
    while (i < vals1.length && i < vals2.length && vals1[i].equals(vals2[i])) {
      i++;
    }
    if (i < vals1.length && i < vals2.length) {
      int diff = Integer.valueOf(vals1[i]).compareTo(Integer.valueOf(vals2[i]));
      return Integer.signum(diff);
    }
    return Integer.signum(vals1.length - vals2.length);
  }

  static boolean isGranular(Map<String, Version> versions) {
    SortedSet<String> firebaseVersions = new TreeSet<String>(GoogleServicesPlugin.&versionCompare)
    for (Version version : versions.values()) {
      firebaseVersions.add(version.getTrimmedVersion())
    }
    firebaseVersions.add(GRANULAR_BUILD_VERSION)
    if (firebaseVersions.first().equals(GRANULAR_BUILD_VERSION)) {
      return true;
    }
    if (firebaseVersions.last().equals(GRANULAR_BUILD_VERSION)) {
      return false;
    }
    throw new GradleException("All firebase libraries must be either above or below $GRANULAR_BUILD_VERSION")
  }

  @groovy.transform.Immutable static class Version {
    String rawVersion, trimmedVersion
  }
}
