/*
 * Copyright 2008 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.google.gwt.user.client.ui;

import java.util.EventListener;

/**
 * Event listener interface for keyboard events.
 */
public interface KeyboardListener extends EventListener {

  int KEY_ALT = 18;
  int KEY_BACKSPACE = 8;
  int KEY_CTRL = 17;
  int KEY_DELETE = 46;
  int KEY_DOWN = 40;
  int KEY_END = 35;
  int KEY_ENTER = 13;
  int KEY_ESCAPE = 27;
  int KEY_HOME = 36;
  int KEY_LEFT = 37;
  int KEY_PAGEDOWN = 34;
  int KEY_PAGEUP = 33;
  int KEY_RIGHT = 39;
  int KEY_SHIFT = 16;
  int KEY_TAB = 9;
  int KEY_UP = 38;

  int MODIFIER_ALT = 4;
  int MODIFIER_CTRL = 2;
  int MODIFIER_META = 8;
  int MODIFIER_SHIFT = 1;

  /**
   * Fired when the user depresses a physical key.
   * 
   * @param sender the widget that was focused when the event occurred.
   * @param keyCode the physical key that was depressed. Constants for this
   *          value are defined in this interface with the KEY prefix.
   * @param modifiers the modifier keys pressed at when the event occurred. This
   *          value is a combination of the bits defined by
   *          {@link KeyboardListener#MODIFIER_SHIFT},
   *          {@link KeyboardListener#MODIFIER_CTRL}, and
   *          {@link KeyboardListener#MODIFIER_ALT}.
   */
  void onKeyDown(Widget sender, char keyCode, int modifiers);

  /**
   * Fired when a keyboard action generates a character. This occurs after
   * onKeyDown and onKeyUp are fired for the physical key that was pressed.
   * 
   * <p>
   * It should be noted that many browsers do not generate keypress events for
   * non-printing keyCode values, such as {@link KeyboardListener#KEY_ENTER} or
   * arrow keys. These keyCodes can be reliably captured either with
   * {@link KeyboardListener#onKeyDown(Widget, char, int)} or
   * {@link KeyboardListener#onKeyUp(Widget, char, int)}.
   * </p>
   * 
   * @param sender the widget that was focused when the event occurred.
   * @param keyCode the Unicode character that was generated by the keyboard
   *          action.
   * @param modifiers the modifier keys pressed at when the event occurred. This
   *          value is a combination of the bits defined by
   *          {@link KeyboardListener#MODIFIER_SHIFT},
   *          {@link KeyboardListener#MODIFIER_CTRL}, and
   *          {@link KeyboardListener#MODIFIER_ALT}.
   */
  void onKeyPress(Widget sender, char keyCode, int modifiers);

  /**
   * Fired when the user releases a physical key.
   * 
   * @param sender the widget that was focused when the event occurred.
   * @param keyCode the physical key that was released. Constants for this value
   *          are defined in this interface with the KEY prefix.
   * @param modifiers the modifier keys pressed at when the event occurred. This
   *          value is a combination of the bits defined by
   *          {@link KeyboardListener#MODIFIER_SHIFT},
   *          {@link KeyboardListener#MODIFIER_CTRL}, and
   *          {@link KeyboardListener#MODIFIER_ALT}.
   */
  void onKeyUp(Widget sender, char keyCode, int modifiers);
}
