// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.

package com.microsoft.aad.msal4j;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.Map;
import java.util.Objects;

/**
 * Contains information about outgoing HTTP request. Should be adapted to HTTP request for HTTP
 * client of choice
 */
public class HttpRequest {

    /**
     * {@link HttpMethod}
     */
    private HttpMethod httpMethod;

    /**
     * HTTP request url
     */
    private URL url;

    /**
     * HTTP request headers
     */
    private Map<String, String> headers;

    /**
     * HTTP request body
     */
    private String body;

    HttpRequest(HttpMethod httpMethod, String url) {
        this.httpMethod = httpMethod;
        this.url = createUrlFromString(url);
    }

    HttpRequest(HttpMethod httpMethod, String url, Map<String, String> headers) {
        this.httpMethod = httpMethod;
        this.url = createUrlFromString(url);
        this.headers = headers;
    }

    HttpRequest(HttpMethod httpMethod, String url, String body) {
        this.httpMethod = httpMethod;
        this.url = createUrlFromString(url);
        this.body = body;
    }

    HttpRequest(HttpMethod httpMethod,
                String url, Map<String, String> headers,
                String body) {
        this.httpMethod = httpMethod;
        this.url = createUrlFromString(url);
        this.headers = headers;
        this.body = body;
    }

    /**
     * @param headerName Name of HTTP header name
     * @return Value of HTTP header
     */
    public String headerValue(String headerName) {

        if (headerName == null || headers == null) {
            return null;
        }

        return headers.get(headerName);
    }

    private URL createUrlFromString(String stringUrl) {
        URL url;
        try {
            url = new URL(stringUrl);
        } catch (MalformedURLException e) {
            throw new MsalClientException(e);
        }

        return url;
    }

    public HttpMethod httpMethod() {
        return this.httpMethod;
    }

    public URL url() {
        return this.url;
    }

    public Map<String, String> headers() {
        return this.headers;
    }

    public String body() {
        return this.body;
    }

    //These methods are based on those generated by Lombok's @EqualsAndHashCode annotation.
    //They have the same functionality as the generated methods, but were refactored for readability.
    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof HttpRequest)) return false;

        HttpRequest other = (HttpRequest) o;
        if (!other.canEqual(this)) return false;

        if (!Objects.equals(httpMethod(), other.httpMethod())) return false;
        if (!Objects.equals(url(), other.url())) return false;
        if (!Objects.equals(headers(), other.headers())) return false;
        return Objects.equals(body(), other.body());
    }

    protected boolean canEqual(Object other) {
        return other instanceof HttpRequest;
    }

    @Override
    public int hashCode() {
        int result = 1;
        result = result * 59 + (this.httpMethod == null ? 43 : this.httpMethod.hashCode());
        result = result * 59 + (this.url == null ? 43 : this.url.hashCode());
        result = result * 59 + (this.headers == null ? 43 : this.headers.hashCode());
        result = result * 59 + (this.body == null ? 43 : this.body.hashCode());
        return result;
    }
}
