/*
 * Decompiled with CFR 0.152.
 */
package com.microsoft.rest.v2.policy;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.microsoft.rest.v2.http.BufferedHttpResponse;
import com.microsoft.rest.v2.http.HttpHeader;
import com.microsoft.rest.v2.http.HttpHeaders;
import com.microsoft.rest.v2.http.HttpRequest;
import com.microsoft.rest.v2.http.HttpResponse;
import com.microsoft.rest.v2.policy.HttpLogDetailLevel;
import com.microsoft.rest.v2.policy.RequestPolicy;
import com.microsoft.rest.v2.policy.RequestPolicyFactory;
import com.microsoft.rest.v2.policy.RequestPolicyOptions;
import com.microsoft.rest.v2.util.FlowableUtil;
import io.netty.handler.codec.http.HttpResponseStatus;
import io.reactivex.Completable;
import io.reactivex.CompletableSource;
import io.reactivex.Single;
import io.reactivex.functions.Consumer;
import io.reactivex.functions.Function;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HttpLoggingPolicyFactory
implements RequestPolicyFactory {
    private static final ObjectMapper PRETTY_PRINTER = new ObjectMapper().enable(SerializationFeature.INDENT_OUTPUT);
    private final HttpLogDetailLevel detailLevel;
    private final boolean prettyPrintJSON;

    public HttpLoggingPolicyFactory(HttpLogDetailLevel detailLevel) {
        this(detailLevel, false);
    }

    public HttpLoggingPolicyFactory(HttpLogDetailLevel detailLevel, boolean prettyPrintJSON) {
        this.detailLevel = detailLevel;
        this.prettyPrintJSON = prettyPrintJSON;
    }

    @Override
    public RequestPolicy create(RequestPolicy next, RequestPolicyOptions options) {
        return new LoggingPolicy(next);
    }

    private final class LoggingPolicy
    implements RequestPolicy {
        private static final int MAX_BODY_LOG_SIZE = 16384;
        private final RequestPolicy next;

        private LoggingPolicy(RequestPolicy next) {
            this.next = next;
        }

        private void log(Logger logger, String s) {
            logger.info(s);
        }

        private long getContentLength(HttpHeaders headers) {
            long contentLength = 0L;
            try {
                contentLength = Long.parseLong(headers.value("content-length"));
            }
            catch (NullPointerException | NumberFormatException runtimeException) {
                // empty catch block
            }
            return contentLength;
        }

        @Override
        public Single<HttpResponse> sendAsync(final HttpRequest request) {
            String context = request.callerMethod();
            if (context == null) {
                context = "";
            }
            final Logger logger = LoggerFactory.getLogger((String)context);
            if (HttpLoggingPolicyFactory.this.detailLevel.shouldLogURL()) {
                this.log(logger, String.format("--> %s %s", request.httpMethod(), request.url()));
            }
            if (HttpLoggingPolicyFactory.this.detailLevel.shouldLogHeaders()) {
                for (HttpHeader header : request.headers()) {
                    this.log(logger, header.toString());
                }
            }
            Completable bodyLoggingTask = Completable.complete();
            if (HttpLoggingPolicyFactory.this.detailLevel.shouldLogBody()) {
                if (request.body() == null) {
                    this.log(logger, "(empty body)");
                    this.log(logger, "--> END " + request.httpMethod());
                } else {
                    boolean isHumanReadableContentType = !"application/octet-stream".equalsIgnoreCase(request.headers().value("Content-Type"));
                    final long contentLength = this.getContentLength(request.headers());
                    if (contentLength < 16384L && isHumanReadableContentType) {
                        try {
                            Single<byte[]> collectedBytes = FlowableUtil.collectBytesInArray(request.body());
                            bodyLoggingTask = collectedBytes.flatMapCompletable((Function)new Function<byte[], CompletableSource>(){

                                public CompletableSource apply(byte[] bytes) throws Exception {
                                    String bodyString = new String(bytes, StandardCharsets.UTF_8);
                                    bodyString = LoggingPolicy.this.prettyPrintIfNeeded(logger, request.headers().value("Content-Type"), bodyString);
                                    LoggingPolicy.this.log(logger, String.format("%s-byte body:\n%s", contentLength, bodyString));
                                    LoggingPolicy.this.log(logger, "--> END " + request.httpMethod());
                                    return Completable.complete();
                                }
                            });
                        }
                        catch (Exception e) {
                            bodyLoggingTask = Completable.error((Throwable)e);
                        }
                    } else {
                        this.log(logger, contentLength + "-byte body: (content not logged)");
                        this.log(logger, "--> END " + request.httpMethod());
                    }
                }
            }
            final long startNs = System.nanoTime();
            return bodyLoggingTask.andThen(this.next.sendAsync(request)).flatMap((Function)new Function<HttpResponse, Single<HttpResponse>>(){

                public Single<HttpResponse> apply(HttpResponse httpResponse) {
                    long tookMs = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startNs);
                    return LoggingPolicy.this.logResponse(logger, httpResponse, request.url(), tookMs);
                }
            }).doOnError((Consumer)new Consumer<Throwable>(){

                public void accept(Throwable throwable) {
                    LoggingPolicy.this.log(logger, "<-- HTTP FAILED: " + throwable);
                }
            });
        }

        private Single<HttpResponse> logResponse(final Logger logger, HttpResponse response, URL url, long tookMs) {
            String contentLengthString = response.headerValue("Content-Length");
            String bodySize = contentLengthString == null || contentLengthString.isEmpty() ? "unknown-length" : contentLengthString + "-byte";
            HttpResponseStatus responseStatus = HttpResponseStatus.valueOf((int)response.statusCode());
            if (HttpLoggingPolicyFactory.this.detailLevel.shouldLogURL()) {
                this.log(logger, String.format("<-- %s %s %s (%s ms, %s body)", response.statusCode(), responseStatus.reasonPhrase(), url, tookMs, bodySize));
            }
            if (HttpLoggingPolicyFactory.this.detailLevel.shouldLogHeaders()) {
                for (HttpHeader header : response.headers()) {
                    this.log(logger, header.toString());
                }
            }
            if (HttpLoggingPolicyFactory.this.detailLevel.shouldLogBody()) {
                long contentLength = this.getContentLength(response.headers());
                final String contentTypeHeader = response.headerValue("Content-Type");
                if (!(contentTypeHeader != null && "application/octet-stream".equalsIgnoreCase(contentTypeHeader) || contentLength == 0L || contentLength >= 16384L)) {
                    final BufferedHttpResponse bufferedResponse = response.buffer();
                    return ((HttpResponse)bufferedResponse).bodyAsString().map((Function)new Function<String, HttpResponse>(){

                        public HttpResponse apply(String s) {
                            s = LoggingPolicy.this.prettyPrintIfNeeded(logger, contentTypeHeader, s);
                            LoggingPolicy.this.log(logger, "Response body:\n" + s);
                            LoggingPolicy.this.log(logger, "<-- END HTTP");
                            return bufferedResponse;
                        }
                    });
                }
                this.log(logger, "(body content not logged)");
                this.log(logger, "<-- END HTTP");
            } else {
                this.log(logger, "<-- END HTTP");
            }
            return Single.just((Object)response);
        }

        private String prettyPrintIfNeeded(Logger logger, String contentType, String body) {
            String result = body;
            if (HttpLoggingPolicyFactory.this.prettyPrintJSON && contentType != null && (contentType.startsWith("application/json") || contentType.startsWith("text/json"))) {
                try {
                    JsonNode deserialized = PRETTY_PRINTER.readTree(body);
                    result = PRETTY_PRINTER.writeValueAsString((Object)deserialized);
                }
                catch (Exception e) {
                    this.log(logger, "Failed to pretty print JSON: " + e.getMessage());
                }
            }
            return result;
        }
    }
}

