/**
 * Copyright 2013 Netflix, Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package rx.subjects;

import static org.junit.Assert.*;
import rx.Observable;
import rx.Observer;
import rx.Subscription;
import rx.util.functions.Action1;
import rx.util.functions.Func0;

/* package */class UnsubscribeTester {

    /*
     * This is purposefully package-only so it does not leak into the public API outside of this package.
     * 
     * This package is implementation details and not part of the Javadocs and thus can change without breaking backwards compatibility.
     * 
     * benjchristensen => I'm procrastinating the decision of where and how these types of classes (see rx.operators.OperatorTester) should exist.
     * If they are only for internal implementations then I don't want them as part of the API.
     * If they are truly useful for everyone to use then an "rx.testing" package may make sense.
     */

    private boolean isDone = false;
    private Subscription subscription;

    public UnsubscribeTester() {
    }

    /**
     * Tests the unsubscription semantics of an observable.
     * 
     * @param provider
     *            Function that when called provides an instance of the observable being tested
     * @param generateOnCompleted
     *            Causes an observer generated by @param provider to generate an onCompleted event. Null to not test onCompleted.
     * @param generateOnError
     *            Causes an observer generated by @param provider to generate an onError event. Null to not test onError.
     * @param generateOnNext
     *            Causes an observer generated by @param provider to generate an onNext event. Null to not test onNext.
     * @param <T>
     *            The type of object passed by the Observable
     */
    public static <T, O extends Observable<T>> void test(Func0<? extends O> provider, Action1<? super O> generateOnCompleted, Action1<? super O> generateOnError, Action1<? super O> generateOnNext)
    {
        if (generateOnCompleted != null) {
            O observable = provider.call();
            UnsubscribeTester tester1 = createOnCompleted(observable);
            UnsubscribeTester tester2 = createOnCompleted(observable);
            generateOnCompleted.call(observable);
            tester1.assertPassed();
            tester2.assertPassed();
        }
        if (generateOnError != null) {
            O observable = provider.call();
            UnsubscribeTester tester1 = createOnError(observable);
            UnsubscribeTester tester2 = createOnError(observable);
            generateOnError.call(observable);
            tester1.assertPassed();
            tester2.assertPassed();
        }
        if (generateOnNext != null) {
            O observable = provider.call();
            UnsubscribeTester tester1 = createOnNext(observable);
            UnsubscribeTester tester2 = createOnNext(observable);
            generateOnNext.call(observable);
            tester1.assertPassed();
            tester2.assertPassed();
        }
    }

    private static <T> UnsubscribeTester createOnCompleted(Observable<T> observable)
    {
        final UnsubscribeTester test = new UnsubscribeTester();
        test.setSubscription(observable.subscribe(new Observer<T>()
        {
            @Override
            public void onCompleted()
            {
                test.doUnsubscribe("onCompleted");
            }

            @Override
            public void onError(Throwable e)
            {
                test.gotEvent("onError");
            }

            @Override
            public void onNext(T args)
            {
                test.gotEvent("onNext");
            }
        }));
        return test;
    }

    private static <T> UnsubscribeTester createOnError(Observable<T> observable)
    {
        final UnsubscribeTester test = new UnsubscribeTester();
        test.setSubscription(observable.subscribe(new Observer<T>()
        {
            @Override
            public void onCompleted()
            {
                test.gotEvent("onCompleted");
            }

            @Override
            public void onError(Throwable e)
            {
                test.doUnsubscribe("onError");
            }

            @Override
            public void onNext(T args)
            {
                test.gotEvent("onNext");
            }
        }));
        return test;
    }

    private static <T> UnsubscribeTester createOnNext(Observable<T> observable)
    {
        final UnsubscribeTester test = new UnsubscribeTester();
        test.setSubscription(observable.subscribe(new Observer<T>()
        {
            @Override
            public void onCompleted()
            {
                test.gotEvent("onCompleted");
            }

            @Override
            public void onError(Throwable e)
            {
                test.gotEvent("onError");
            }

            @Override
            public void onNext(T args)
            {
                test.doUnsubscribe("onNext");
            }
        }));
        return test;
    }

    private void setSubscription(Subscription subscription)
    {
        this.subscription = subscription;
    }

    private void gotEvent(String event)
    {
        assertFalse("received " + event + " after unsubscribe", isDone);
    }

    private void doUnsubscribe(String event)
    {
        gotEvent(event);
        if (subscription != null) {
            isDone = true;
            subscription.unsubscribe();
        }
    }

    private void assertPassed()
    {
        assertTrue("expected notification was received", isDone);
    }
}
