/*
 * Copyright (C) 2015 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.basics.value;

import java.io.Serializable;
import java.lang.invoke.MethodHandles;

import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.TypedMetaBean;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.light.LightMetaBean;

import com.opengamma.strata.collect.array.DoubleArray;

/**
 * A value and its derivatives.
 * <p>
 * This defines a standard way to return a value and its derivatives to certain inputs.
 * It is in particular used as a return object for Algorithmic Differentiation versions of some functions.
 */
@BeanDefinition(style = "light")
public final class ValueDerivatives
    implements ImmutableBean, Serializable {

  /**
   * The value of the variable.
   */
  @PropertyDefinition
  private final double value;
  /**
   * The derivatives of the variable with respect to some inputs.
   */
  @PropertyDefinition(validate = "notNull")
  private final DoubleArray derivatives;

  //-------------------------------------------------------------------------
  /**
   * Obtains an instance from a value and array of derivatives.
   * 
   * @param value  the value
   * @param derivatives  the derivatives of the value
   * @return the object
   */
  public static ValueDerivatives of(double value, DoubleArray derivatives) {
    return new ValueDerivatives(value, derivatives);
  }

  //-------------------------------------------------------------------------
  /**
   * Gets the derivative of the variable with respect to an input.
   * 
   * @param index  the zero-based derivative to obtain
   * @return the derivative
   * @throws IndexOutOfBoundsException if the index is invalid
   */
  public double getDerivative(int index) {
    return derivatives.get(index);
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code ValueDerivatives}.
   */
  private static final TypedMetaBean<ValueDerivatives> META_BEAN =
      LightMetaBean.of(
          ValueDerivatives.class,
          MethodHandles.lookup(),
          new String[] {
              "value",
              "derivatives"},
          new Object[0]);

  /**
   * The meta-bean for {@code ValueDerivatives}.
   * @return the meta-bean, not null
   */
  public static TypedMetaBean<ValueDerivatives> meta() {
    return META_BEAN;
  }

  static {
    MetaBean.register(META_BEAN);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  private ValueDerivatives(
      double value,
      DoubleArray derivatives) {
    JodaBeanUtils.notNull(derivatives, "derivatives");
    this.value = value;
    this.derivatives = derivatives;
  }

  @Override
  public TypedMetaBean<ValueDerivatives> metaBean() {
    return META_BEAN;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the value of the variable.
   * @return the value of the property
   */
  public double getValue() {
    return value;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the derivatives of the variable with respect to some inputs.
   * @return the value of the property, not null
   */
  public DoubleArray getDerivatives() {
    return derivatives;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      ValueDerivatives other = (ValueDerivatives) obj;
      return JodaBeanUtils.equal(value, other.value) &&
          JodaBeanUtils.equal(derivatives, other.derivatives);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(value);
    hash = hash * 31 + JodaBeanUtils.hashCode(derivatives);
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(96);
    buf.append("ValueDerivatives{");
    buf.append("value").append('=').append(JodaBeanUtils.toString(value)).append(',').append(' ');
    buf.append("derivatives").append('=').append(JodaBeanUtils.toString(derivatives));
    buf.append('}');
    return buf.toString();
  }

  //-------------------------- AUTOGENERATED END --------------------------
}
