/*
 * Copyright (C) 2016 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.data;

import java.io.Serializable;
import java.lang.invoke.MethodHandles;

import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.TypedMetaBean;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.light.LightMetaBean;

import com.opengamma.strata.basics.currency.FxMatrix;

/**
 * Identifies the market data for an FX matrix.
 */
@BeanDefinition(style = "light", cacheHashCode = true)
public final class FxMatrixId
    implements MarketDataId<FxMatrix>, ImmutableBean, Serializable {

  /**
   * The standard identifier.
   */
  private static final FxMatrixId STANDARD = new FxMatrixId(ObservableSource.NONE);

  /**
   * The source of observable market data.
   * This is used when looking up the underlying market quotes for the rate.
   */
  @PropertyDefinition(validate = "notNull")
  private final ObservableSource observableSource;

  //-------------------------------------------------------------------------
  /**
   * Obtains an instance representing an FX matrix.
   *
   * @return an identifier for the FX matrix
   */
  public static FxMatrixId standard() {
    return STANDARD;
  }

  /**
   * Obtains an instance representing an FX matrix, specifying the source.
   *
   * @param observableSource  the source of the observable market data used to create the rate
   * @return an identifier for the FX matrix
   */
  public static FxMatrixId of(ObservableSource observableSource) {
    return new FxMatrixId(observableSource);
  }

  //-------------------------------------------------------------------------
  @Override
  public Class<FxMatrix> getMarketDataType() {
    return FxMatrix.class;
  }

  @Override
  public String toString() {
    return "FxMatrixId" + (observableSource.equals(ObservableSource.NONE) ? "" : ":" + observableSource);
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code FxMatrixId}.
   */
  private static final TypedMetaBean<FxMatrixId> META_BEAN =
      LightMetaBean.of(
          FxMatrixId.class,
          MethodHandles.lookup(),
          new String[] {
              "observableSource"},
          new Object[0]);

  /**
   * The meta-bean for {@code FxMatrixId}.
   * @return the meta-bean, not null
   */
  public static TypedMetaBean<FxMatrixId> meta() {
    return META_BEAN;
  }

  static {
    MetaBean.register(META_BEAN);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  /**
   * The cached hash code, using the racy single-check idiom.
   */
  private transient int cacheHashCode;

  private FxMatrixId(
      ObservableSource observableSource) {
    JodaBeanUtils.notNull(observableSource, "observableSource");
    this.observableSource = observableSource;
  }

  @Override
  public TypedMetaBean<FxMatrixId> metaBean() {
    return META_BEAN;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the source of observable market data.
   * This is used when looking up the underlying market quotes for the rate.
   * @return the value of the property, not null
   */
  public ObservableSource getObservableSource() {
    return observableSource;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      FxMatrixId other = (FxMatrixId) obj;
      return JodaBeanUtils.equal(observableSource, other.observableSource);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = cacheHashCode;
    if (hash == 0) {
      hash = getClass().hashCode();
      hash = hash * 31 + JodaBeanUtils.hashCode(observableSource);
      cacheHashCode = hash;
    }
    return hash;
  }

  //-------------------------- AUTOGENERATED END --------------------------
}
