/*
 * Copyright (C) 2015 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.data;

import java.io.Serializable;
import java.lang.invoke.MethodHandles;

import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.TypedMetaBean;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.ImmutableConstructor;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.light.LightMetaBean;

import com.opengamma.strata.basics.currency.Currency;
import com.opengamma.strata.basics.currency.CurrencyPair;
import com.opengamma.strata.basics.currency.FxRate;
import com.opengamma.strata.collect.ArgChecker;

/**
 * Identifies the market data for an FX rate.
 * <p>
 * The currency pair in a rate ID is always the market convention currency pair. If an ID is
 * created using a non-conventional pair, the pair is inverted. This has no effect on the
 * {@link FxRate} identified by the ID as it can handle both currency pairs that can be
 * created from the two currencies.
 */
@BeanDefinition(style = "light", cacheHashCode = true)
public final class FxRateId
    implements MarketDataId<FxRate>, ImmutableBean, Serializable {

  /**
   * The currency pair that is required.
   * For example, 'GBP/USD'.
   */
  @PropertyDefinition(validate = "notNull")
  private final CurrencyPair pair;
  /**
   * The source of observable market data.
   * This is used when looking up the underlying market quotes for the rate.
   */
  @PropertyDefinition(validate = "notNull")
  private final ObservableSource observableSource;

  //-------------------------------------------------------------------------
  /**
   * Obtains an instance representing the FX rate for a currency pair.
   *
   * @param currencyPair  a currency pair
   * @return an ID for the FX rate for the currency pair
   */
  public static FxRateId of(CurrencyPair currencyPair) {
    return new FxRateId(currencyPair, ObservableSource.NONE);
  }

  /**
   * Obtains an instance representing the FX rate for a currency pair.
   *
   * @param base  the base currency of the pair
   * @param counter  the counter currency of the pair
   * @return an ID for the FX rate for the currency pair
   */
  public static FxRateId of(Currency base, Currency counter) {
    return new FxRateId(CurrencyPair.of(base, counter), ObservableSource.NONE);
  }

  /**
   * Obtains an instance representing the FX rate for a currency pair, specifying the source.
   *
   * @param currencyPair  a currency pair
   * @param observableSource  the source of the observable market data used to create the rate
   * @return an ID for the FX rate for the currency pair
   */
  public static FxRateId of(CurrencyPair currencyPair, ObservableSource observableSource) {
    return new FxRateId(currencyPair, observableSource);
  }

  /**
   * Obtains an instance representing the FX rate for a currency pair, specifying the source.
   *
   * @param base  the base currency of the pair
   * @param counter  the counter currency of the pair
   * @param observableSource  the source of the observable market data used to create the rate
   * @return an ID for the FX rate for the currency pair
   */
  public static FxRateId of(Currency base, Currency counter, ObservableSource observableSource) {
    return new FxRateId(CurrencyPair.of(base, counter), observableSource);
  }

  @ImmutableConstructor
  private FxRateId(CurrencyPair currencyPair, ObservableSource observableSource) {
    ArgChecker.notNull(currencyPair, "currencyPair");
    ArgChecker.notNull(observableSource, "observableSource");
    this.pair = currencyPair.toConventional();
    this.observableSource = observableSource;
  }

  //-------------------------------------------------------------------------
  @Override
  public Class<FxRate> getMarketDataType() {
    return FxRate.class;
  }

  @Override
  public String toString() {
    return new StringBuilder(32)
        .append("FxRateId:")
        .append(pair)
        .append(observableSource.equals(ObservableSource.NONE) ? "" : "/" + observableSource)
        .toString();
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code FxRateId}.
   */
  private static final TypedMetaBean<FxRateId> META_BEAN =
      LightMetaBean.of(
          FxRateId.class,
          MethodHandles.lookup(),
          new String[] {
              "pair",
              "observableSource"},
          new Object[0]);

  /**
   * The meta-bean for {@code FxRateId}.
   * @return the meta-bean, not null
   */
  public static TypedMetaBean<FxRateId> meta() {
    return META_BEAN;
  }

  static {
    MetaBean.register(META_BEAN);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  /**
   * The cached hash code, using the racy single-check idiom.
   */
  private transient int cacheHashCode;

  @Override
  public TypedMetaBean<FxRateId> metaBean() {
    return META_BEAN;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the currency pair that is required.
   * For example, 'GBP/USD'.
   * @return the value of the property, not null
   */
  public CurrencyPair getPair() {
    return pair;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the source of observable market data.
   * This is used when looking up the underlying market quotes for the rate.
   * @return the value of the property, not null
   */
  public ObservableSource getObservableSource() {
    return observableSource;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      FxRateId other = (FxRateId) obj;
      return JodaBeanUtils.equal(pair, other.pair) &&
          JodaBeanUtils.equal(observableSource, other.observableSource);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = cacheHashCode;
    if (hash == 0) {
      hash = getClass().hashCode();
      hash = hash * 31 + JodaBeanUtils.hashCode(pair);
      hash = hash * 31 + JodaBeanUtils.hashCode(observableSource);
      cacheHashCode = hash;
    }
    return hash;
  }

  //-------------------------- AUTOGENERATED END --------------------------
}
