/*
 * Copyright (C) 2016 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.data.scenario;

import java.io.Serializable;
import java.time.LocalDate;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Optional;
import java.util.Set;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import org.joda.beans.Bean;
import org.joda.beans.BeanBuilder;
import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.MetaProperty;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.direct.DirectMetaBean;
import org.joda.beans.impl.direct.DirectMetaProperty;
import org.joda.beans.impl.direct.DirectMetaPropertyMap;
import org.joda.beans.impl.direct.DirectPrivateBeanBuilder;

import com.google.common.base.Preconditions;
import com.opengamma.strata.collect.ArgChecker;
import com.opengamma.strata.collect.timeseries.LocalDateDoubleTimeSeries;
import com.opengamma.strata.data.MarketData;
import com.opengamma.strata.data.MarketDataId;
import com.opengamma.strata.data.MarketDataName;
import com.opengamma.strata.data.ObservableId;

/**
 * Market data across one or more scenarios based on a single set of market data.
 * <p>
 * This implementation of {@link ScenarioMarketData} returns the same {@link MarketData}
 * instance for each scenario.
 */
@BeanDefinition(builderScope = "private", constructorScope = "package")
final class RepeatedScenarioMarketData
    implements ScenarioMarketData, ImmutableBean, Serializable {

  /**
   * The number of scenarios.
   */
  @PropertyDefinition(validate = "ArgChecker.notNegativeOrZero", overrideGet = true)
  private final int scenarioCount;
  /**
   * The underlying market data.
   */
  @PropertyDefinition(validate = "notNull")
  private final MarketData underlying;

  //-------------------------------------------------------------------------
  /**
   * Obtains an instance from a valuation date, map of values and time-series.
   * <p>
   * The valuation date and map of values must have the same number of scenarios.
   * 
   * @param scenarioCount  the number of scenarios
   * @param marketData  the single set of market data
   * @return the scenario market data
   */
  public static RepeatedScenarioMarketData of(int scenarioCount, MarketData marketData) {
    return new RepeatedScenarioMarketData(scenarioCount, marketData);
  }

  //-------------------------------------------------------------------------
  @Override
  public MarketDataBox<LocalDate> getValuationDate() {
    return MarketDataBox.ofSingleValue(underlying.getValuationDate());
  }

  @Override
  public Stream<MarketData> scenarios() {
    return IntStream.range(0, getScenarioCount())
        .mapToObj(scenarioIndex -> underlying);
  }

  @Override
  public MarketData scenario(int scenarioIndex) {
    Preconditions.checkElementIndex(scenarioIndex, scenarioCount, "scenarioIndex");
    return underlying;
  }

  //-------------------------------------------------------------------------
  @Override
  public boolean containsValue(MarketDataId<?> id) {
    return underlying.containsValue(id);
  }

  @SuppressWarnings("unchecked")
  @Override
  public <T> MarketDataBox<T> getValue(MarketDataId<T> id) {
    return MarketDataBox.ofSingleValue(underlying.getValue(id));
  }

  @SuppressWarnings("unchecked")
  @Override
  public <T> Optional<MarketDataBox<T>> findValue(MarketDataId<T> id) {
    return underlying.findValue(id).map(v -> MarketDataBox.ofSingleValue(v));
  }

  @Override
  public Set<MarketDataId<?>> getIds() {
    return underlying.getIds();
  }

  @Override
  @SuppressWarnings("unchecked")
  public <T> Set<MarketDataId<T>> findIds(MarketDataName<T> name) {
    return underlying.findIds(name);
  }

  @Override
  public Set<ObservableId> getTimeSeriesIds() {
    return underlying.getTimeSeriesIds();
  }

  @Override
  public LocalDateDoubleTimeSeries getTimeSeries(ObservableId id) {
    return underlying.getTimeSeries(id);
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code RepeatedScenarioMarketData}.
   * @return the meta-bean, not null
   */
  public static RepeatedScenarioMarketData.Meta meta() {
    return RepeatedScenarioMarketData.Meta.INSTANCE;
  }

  static {
    MetaBean.register(RepeatedScenarioMarketData.Meta.INSTANCE);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  /**
   * Creates an instance.
   * @param scenarioCount  the value of the property
   * @param underlying  the value of the property, not null
   */
  RepeatedScenarioMarketData(
      int scenarioCount,
      MarketData underlying) {
    ArgChecker.notNegativeOrZero(scenarioCount, "scenarioCount");
    JodaBeanUtils.notNull(underlying, "underlying");
    this.scenarioCount = scenarioCount;
    this.underlying = underlying;
  }

  @Override
  public RepeatedScenarioMarketData.Meta metaBean() {
    return RepeatedScenarioMarketData.Meta.INSTANCE;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the number of scenarios.
   * @return the value of the property
   */
  @Override
  public int getScenarioCount() {
    return scenarioCount;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the underlying market data.
   * @return the value of the property, not null
   */
  public MarketData getUnderlying() {
    return underlying;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      RepeatedScenarioMarketData other = (RepeatedScenarioMarketData) obj;
      return (scenarioCount == other.scenarioCount) &&
          JodaBeanUtils.equal(underlying, other.underlying);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(scenarioCount);
    hash = hash * 31 + JodaBeanUtils.hashCode(underlying);
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(96);
    buf.append("RepeatedScenarioMarketData{");
    buf.append("scenarioCount").append('=').append(JodaBeanUtils.toString(scenarioCount)).append(',').append(' ');
    buf.append("underlying").append('=').append(JodaBeanUtils.toString(underlying));
    buf.append('}');
    return buf.toString();
  }

  //-----------------------------------------------------------------------
  /**
   * The meta-bean for {@code RepeatedScenarioMarketData}.
   */
  static final class Meta extends DirectMetaBean {
    /**
     * The singleton instance of the meta-bean.
     */
    static final Meta INSTANCE = new Meta();

    /**
     * The meta-property for the {@code scenarioCount} property.
     */
    private final MetaProperty<Integer> scenarioCount = DirectMetaProperty.ofImmutable(
        this, "scenarioCount", RepeatedScenarioMarketData.class, Integer.TYPE);
    /**
     * The meta-property for the {@code underlying} property.
     */
    private final MetaProperty<MarketData> underlying = DirectMetaProperty.ofImmutable(
        this, "underlying", RepeatedScenarioMarketData.class, MarketData.class);
    /**
     * The meta-properties.
     */
    private final Map<String, MetaProperty<?>> metaPropertyMap$ = new DirectMetaPropertyMap(
        this, null,
        "scenarioCount",
        "underlying");

    /**
     * Restricted constructor.
     */
    private Meta() {
    }

    @Override
    protected MetaProperty<?> metaPropertyGet(String propertyName) {
      switch (propertyName.hashCode()) {
        case -1203198113:  // scenarioCount
          return scenarioCount;
        case -1770633379:  // underlying
          return underlying;
      }
      return super.metaPropertyGet(propertyName);
    }

    @Override
    public BeanBuilder<? extends RepeatedScenarioMarketData> builder() {
      return new RepeatedScenarioMarketData.Builder();
    }

    @Override
    public Class<? extends RepeatedScenarioMarketData> beanType() {
      return RepeatedScenarioMarketData.class;
    }

    @Override
    public Map<String, MetaProperty<?>> metaPropertyMap() {
      return metaPropertyMap$;
    }

    //-----------------------------------------------------------------------
    /**
     * The meta-property for the {@code scenarioCount} property.
     * @return the meta-property, not null
     */
    public MetaProperty<Integer> scenarioCount() {
      return scenarioCount;
    }

    /**
     * The meta-property for the {@code underlying} property.
     * @return the meta-property, not null
     */
    public MetaProperty<MarketData> underlying() {
      return underlying;
    }

    //-----------------------------------------------------------------------
    @Override
    protected Object propertyGet(Bean bean, String propertyName, boolean quiet) {
      switch (propertyName.hashCode()) {
        case -1203198113:  // scenarioCount
          return ((RepeatedScenarioMarketData) bean).getScenarioCount();
        case -1770633379:  // underlying
          return ((RepeatedScenarioMarketData) bean).getUnderlying();
      }
      return super.propertyGet(bean, propertyName, quiet);
    }

    @Override
    protected void propertySet(Bean bean, String propertyName, Object newValue, boolean quiet) {
      metaProperty(propertyName);
      if (quiet) {
        return;
      }
      throw new UnsupportedOperationException("Property cannot be written: " + propertyName);
    }

  }

  //-----------------------------------------------------------------------
  /**
   * The bean-builder for {@code RepeatedScenarioMarketData}.
   */
  private static final class Builder extends DirectPrivateBeanBuilder<RepeatedScenarioMarketData> {

    private int scenarioCount;
    private MarketData underlying;

    /**
     * Restricted constructor.
     */
    private Builder() {
    }

    //-----------------------------------------------------------------------
    @Override
    public Object get(String propertyName) {
      switch (propertyName.hashCode()) {
        case -1203198113:  // scenarioCount
          return scenarioCount;
        case -1770633379:  // underlying
          return underlying;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
    }

    @Override
    public Builder set(String propertyName, Object newValue) {
      switch (propertyName.hashCode()) {
        case -1203198113:  // scenarioCount
          this.scenarioCount = (Integer) newValue;
          break;
        case -1770633379:  // underlying
          this.underlying = (MarketData) newValue;
          break;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
      return this;
    }

    @Override
    public RepeatedScenarioMarketData build() {
      return new RepeatedScenarioMarketData(
          scenarioCount,
          underlying);
    }

    //-----------------------------------------------------------------------
    @Override
    public String toString() {
      StringBuilder buf = new StringBuilder(96);
      buf.append("RepeatedScenarioMarketData.Builder{");
      buf.append("scenarioCount").append('=').append(JodaBeanUtils.toString(scenarioCount)).append(',').append(' ');
      buf.append("underlying").append('=').append(JodaBeanUtils.toString(underlying));
      buf.append('}');
      return buf.toString();
    }

  }

  //-------------------------- AUTOGENERATED END --------------------------
}
