/*
 * Copyright (C) 2016 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.data.scenario;

import static com.opengamma.strata.collect.Guavate.toImmutableList;

import java.io.Serializable;
import java.util.Collections;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import org.joda.beans.Bean;
import org.joda.beans.BeanBuilder;
import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.MetaProperty;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.direct.DirectMetaBean;
import org.joda.beans.impl.direct.DirectMetaProperty;
import org.joda.beans.impl.direct.DirectMetaPropertyMap;
import org.joda.beans.impl.direct.DirectPrivateBeanBuilder;

import com.google.common.collect.ImmutableList;
import com.opengamma.strata.basics.currency.Currency;
import com.opengamma.strata.basics.currency.FxConvertible;
import com.opengamma.strata.collect.ArgChecker;
import com.opengamma.strata.collect.Messages;

/**
 * A scenario array holding one value that is valid for all scenarios.
 * <p>
 * This contains a single value where the same value is the result of each scenario.
 * The calculation runner will not attempt to convert the currency of the value.
 * 
 * @param <T>  the type of the result
 */
@BeanDefinition(builderScope = "private")
final class SingleScenarioArray<T>
    implements ScenarioArray<T>, ScenarioFxConvertible<ScenarioArray<?>>, ImmutableBean, Serializable {

  /**
   * The number of scenarios.
   */
  @PropertyDefinition(validate = "notNull", overrideGet = true)
  private final int scenarioCount;
  /**
   * The single value that applies to all scenarios.
   */
  @PropertyDefinition(validate = "notNull")
  private final T value;

  //-------------------------------------------------------------------------
  /**
   * Obtains an instance from a single value and scenario count.
   * <p>
   * The single value is valid for each scenario.
   *
   * @param <T>  the type of the value
   * @param scenarioCount  the number of scenarios
   * @param value  the single value valid for all scenarios
   * @return an instance with the specified value and count
   */
  public static <T> SingleScenarioArray<T> of(int scenarioCount, T value) {
    return new SingleScenarioArray<>(scenarioCount, value);
  }

  //-------------------------------------------------------------------------
  @Override
  public T get(int index) {
    ArgChecker.inRange(index, 0, scenarioCount, "index");
    return value;
  }

  @Override
  public Stream<T> stream() {
    return Collections.nCopies(scenarioCount, value).stream();
  }

  //-------------------------------------------------------------------------
  @Override
  public ScenarioArray<?> convertedTo(Currency reportingCurrency, ScenarioFxRateProvider fxRateProvider) {
    if (value instanceof FxConvertible<?>) {
      FxConvertible<?> convertible = (FxConvertible<?>) value;
      if (fxRateProvider.getScenarioCount() != scenarioCount) {
        throw new IllegalArgumentException(Messages.format(
            "Expected {} FX rates but received {}", scenarioCount, fxRateProvider.getScenarioCount()));
      }
      ImmutableList<Object> converted = IntStream.range(0, scenarioCount)
          .mapToObj(i -> convertible.convertedTo(reportingCurrency, fxRateProvider.fxRateProvider(i)))
          .collect(toImmutableList());
      return DefaultScenarioArray.of(converted);
    }
    return this;
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code SingleScenarioArray}.
   * @return the meta-bean, not null
   */
  @SuppressWarnings("rawtypes")
  public static SingleScenarioArray.Meta meta() {
    return SingleScenarioArray.Meta.INSTANCE;
  }

  /**
   * The meta-bean for {@code SingleScenarioArray}.
   * @param <R>  the bean's generic type
   * @param cls  the bean's generic type
   * @return the meta-bean, not null
   */
  @SuppressWarnings("unchecked")
  public static <R> SingleScenarioArray.Meta<R> metaSingleScenarioArray(Class<R> cls) {
    return SingleScenarioArray.Meta.INSTANCE;
  }

  static {
    MetaBean.register(SingleScenarioArray.Meta.INSTANCE);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  private SingleScenarioArray(
      int scenarioCount,
      T value) {
    JodaBeanUtils.notNull(scenarioCount, "scenarioCount");
    JodaBeanUtils.notNull(value, "value");
    this.scenarioCount = scenarioCount;
    this.value = value;
  }

  @SuppressWarnings("unchecked")
  @Override
  public SingleScenarioArray.Meta<T> metaBean() {
    return SingleScenarioArray.Meta.INSTANCE;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the number of scenarios.
   * @return the value of the property, not null
   */
  @Override
  public int getScenarioCount() {
    return scenarioCount;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the single value that applies to all scenarios.
   * @return the value of the property, not null
   */
  public T getValue() {
    return value;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      SingleScenarioArray<?> other = (SingleScenarioArray<?>) obj;
      return (scenarioCount == other.scenarioCount) &&
          JodaBeanUtils.equal(value, other.value);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(scenarioCount);
    hash = hash * 31 + JodaBeanUtils.hashCode(value);
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(96);
    buf.append("SingleScenarioArray{");
    buf.append("scenarioCount").append('=').append(JodaBeanUtils.toString(scenarioCount)).append(',').append(' ');
    buf.append("value").append('=').append(JodaBeanUtils.toString(value));
    buf.append('}');
    return buf.toString();
  }

  //-----------------------------------------------------------------------
  /**
   * The meta-bean for {@code SingleScenarioArray}.
   * @param <T>  the type
   */
  static final class Meta<T> extends DirectMetaBean {
    /**
     * The singleton instance of the meta-bean.
     */
    @SuppressWarnings("rawtypes")
    static final Meta INSTANCE = new Meta();

    /**
     * The meta-property for the {@code scenarioCount} property.
     */
    private final MetaProperty<Integer> scenarioCount = DirectMetaProperty.ofImmutable(
        this, "scenarioCount", SingleScenarioArray.class, Integer.TYPE);
    /**
     * The meta-property for the {@code value} property.
     */
    @SuppressWarnings({"unchecked", "rawtypes" })
    private final MetaProperty<T> value = (DirectMetaProperty) DirectMetaProperty.ofImmutable(
        this, "value", SingleScenarioArray.class, Object.class);
    /**
     * The meta-properties.
     */
    private final Map<String, MetaProperty<?>> metaPropertyMap$ = new DirectMetaPropertyMap(
        this, null,
        "scenarioCount",
        "value");

    /**
     * Restricted constructor.
     */
    private Meta() {
    }

    @Override
    protected MetaProperty<?> metaPropertyGet(String propertyName) {
      switch (propertyName.hashCode()) {
        case -1203198113:  // scenarioCount
          return scenarioCount;
        case 111972721:  // value
          return value;
      }
      return super.metaPropertyGet(propertyName);
    }

    @Override
    public BeanBuilder<? extends SingleScenarioArray<T>> builder() {
      return new SingleScenarioArray.Builder<>();
    }

    @SuppressWarnings({"unchecked", "rawtypes" })
    @Override
    public Class<? extends SingleScenarioArray<T>> beanType() {
      return (Class) SingleScenarioArray.class;
    }

    @Override
    public Map<String, MetaProperty<?>> metaPropertyMap() {
      return metaPropertyMap$;
    }

    //-----------------------------------------------------------------------
    /**
     * The meta-property for the {@code scenarioCount} property.
     * @return the meta-property, not null
     */
    public MetaProperty<Integer> scenarioCount() {
      return scenarioCount;
    }

    /**
     * The meta-property for the {@code value} property.
     * @return the meta-property, not null
     */
    public MetaProperty<T> value() {
      return value;
    }

    //-----------------------------------------------------------------------
    @Override
    protected Object propertyGet(Bean bean, String propertyName, boolean quiet) {
      switch (propertyName.hashCode()) {
        case -1203198113:  // scenarioCount
          return ((SingleScenarioArray<?>) bean).getScenarioCount();
        case 111972721:  // value
          return ((SingleScenarioArray<?>) bean).getValue();
      }
      return super.propertyGet(bean, propertyName, quiet);
    }

    @Override
    protected void propertySet(Bean bean, String propertyName, Object newValue, boolean quiet) {
      metaProperty(propertyName);
      if (quiet) {
        return;
      }
      throw new UnsupportedOperationException("Property cannot be written: " + propertyName);
    }

  }

  //-----------------------------------------------------------------------
  /**
   * The bean-builder for {@code SingleScenarioArray}.
   * @param <T>  the type
   */
  private static final class Builder<T> extends DirectPrivateBeanBuilder<SingleScenarioArray<T>> {

    private int scenarioCount;
    private T value;

    /**
     * Restricted constructor.
     */
    private Builder() {
    }

    //-----------------------------------------------------------------------
    @Override
    public Object get(String propertyName) {
      switch (propertyName.hashCode()) {
        case -1203198113:  // scenarioCount
          return scenarioCount;
        case 111972721:  // value
          return value;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
    }

    @SuppressWarnings("unchecked")
    @Override
    public Builder<T> set(String propertyName, Object newValue) {
      switch (propertyName.hashCode()) {
        case -1203198113:  // scenarioCount
          this.scenarioCount = (Integer) newValue;
          break;
        case 111972721:  // value
          this.value = (T) newValue;
          break;
        default:
          throw new NoSuchElementException("Unknown property: " + propertyName);
      }
      return this;
    }

    @Override
    public SingleScenarioArray<T> build() {
      return new SingleScenarioArray<>(
          scenarioCount,
          value);
    }

    //-----------------------------------------------------------------------
    @Override
    public String toString() {
      StringBuilder buf = new StringBuilder(96);
      buf.append("SingleScenarioArray.Builder{");
      buf.append("scenarioCount").append('=').append(JodaBeanUtils.toString(scenarioCount)).append(',').append(' ');
      buf.append("value").append('=').append(JodaBeanUtils.toString(value));
      buf.append('}');
      return buf.toString();
    }

  }

  //-------------------------- AUTOGENERATED END --------------------------
}
