/*
 * Copyright (C) 2016 - present by OpenGamma Inc. and the OpenGamma group of companies
 *
 * Please see distribution for license.
 */
package com.opengamma.strata.measure.bond;

import java.io.Serializable;
import java.lang.invoke.MethodHandles;

import org.joda.beans.ImmutableBean;
import org.joda.beans.JodaBeanUtils;
import org.joda.beans.MetaBean;
import org.joda.beans.TypedMetaBean;
import org.joda.beans.gen.BeanDefinition;
import org.joda.beans.gen.ImmutableConstructor;
import org.joda.beans.gen.PropertyDefinition;
import org.joda.beans.impl.light.LightMetaBean;

import com.opengamma.strata.collect.ArgChecker;
import com.opengamma.strata.data.MarketData;
import com.opengamma.strata.pricer.bond.LegalEntityDiscountingProvider;

/**
 * The default market data for products based on repo and issuer curves.
 * <p>
 * This uses a {@link LegalEntityDiscountingMarketDataLookup} to provide a view on {@link MarketData}.
 */
@BeanDefinition(style = "light")
final class DefaultLegalEntityDiscountingMarketData
    implements LegalEntityDiscountingMarketData, ImmutableBean, Serializable {

  /**
   * The lookup.
   */
  @PropertyDefinition(validate = "notNull", overrideGet = true)
  private final LegalEntityDiscountingMarketDataLookup lookup;
  /**
   * The market data.
   */
  @PropertyDefinition(validate = "notNull", overrideGet = true)
  private final MarketData marketData;
  /**
   * The discounting provider.
   */
  private final transient LegalEntityDiscountingProvider discountingProvider;  // derived

  //-------------------------------------------------------------------------
  /**
   * Obtains an instance based on a lookup and market data.
   * <p>
   * The lookup provides the mapping to find the correct repo and issuer curves.
   * The curves are in the market data.
   *
   * @param lookup  the lookup
   * @param marketData  the market data
   * @return the rates market view
   */
  static DefaultLegalEntityDiscountingMarketData of(LegalEntityDiscountingMarketDataLookup lookup, MarketData marketData) {
    return new DefaultLegalEntityDiscountingMarketData(lookup, marketData);
  }

  @ImmutableConstructor
  private DefaultLegalEntityDiscountingMarketData(LegalEntityDiscountingMarketDataLookup lookup, MarketData marketData) {
    this.lookup = ArgChecker.notNull(lookup, "lookup");
    this.marketData = ArgChecker.notNull(marketData, "marketData");
    this.discountingProvider = lookup.discountingProvider(marketData);
  }

  // ensure standard constructor is invoked
  private Object readResolve() {
    return new DefaultLegalEntityDiscountingMarketData(lookup, marketData);
  }

  //-------------------------------------------------------------------------
  @Override
  public LegalEntityDiscountingMarketData withMarketData(MarketData marketData) {
    return DefaultLegalEntityDiscountingMarketData.of(lookup, marketData);
  }

  //-------------------------------------------------------------------------
  @Override
  public LegalEntityDiscountingProvider discountingProvider() {
    return discountingProvider;
  }

  //------------------------- AUTOGENERATED START -------------------------
  /**
   * The meta-bean for {@code DefaultLegalEntityDiscountingMarketData}.
   */
  private static final TypedMetaBean<DefaultLegalEntityDiscountingMarketData> META_BEAN =
      LightMetaBean.of(
          DefaultLegalEntityDiscountingMarketData.class,
          MethodHandles.lookup(),
          new String[] {
              "lookup",
              "marketData"},
          new Object[0]);

  /**
   * The meta-bean for {@code DefaultLegalEntityDiscountingMarketData}.
   * @return the meta-bean, not null
   */
  public static TypedMetaBean<DefaultLegalEntityDiscountingMarketData> meta() {
    return META_BEAN;
  }

  static {
    MetaBean.register(META_BEAN);
  }

  /**
   * The serialization version id.
   */
  private static final long serialVersionUID = 1L;

  @Override
  public TypedMetaBean<DefaultLegalEntityDiscountingMarketData> metaBean() {
    return META_BEAN;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the lookup.
   * @return the value of the property, not null
   */
  @Override
  public LegalEntityDiscountingMarketDataLookup getLookup() {
    return lookup;
  }

  //-----------------------------------------------------------------------
  /**
   * Gets the market data.
   * @return the value of the property, not null
   */
  @Override
  public MarketData getMarketData() {
    return marketData;
  }

  //-----------------------------------------------------------------------
  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj != null && obj.getClass() == this.getClass()) {
      DefaultLegalEntityDiscountingMarketData other = (DefaultLegalEntityDiscountingMarketData) obj;
      return JodaBeanUtils.equal(lookup, other.lookup) &&
          JodaBeanUtils.equal(marketData, other.marketData);
    }
    return false;
  }

  @Override
  public int hashCode() {
    int hash = getClass().hashCode();
    hash = hash * 31 + JodaBeanUtils.hashCode(lookup);
    hash = hash * 31 + JodaBeanUtils.hashCode(marketData);
    return hash;
  }

  @Override
  public String toString() {
    StringBuilder buf = new StringBuilder(96);
    buf.append("DefaultLegalEntityDiscountingMarketData{");
    buf.append("lookup").append('=').append(JodaBeanUtils.toString(lookup)).append(',').append(' ');
    buf.append("marketData").append('=').append(JodaBeanUtils.toString(marketData));
    buf.append('}');
    return buf.toString();
  }

  //-------------------------- AUTOGENERATED END --------------------------
}
