package com.vaadin.copilot;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.URI;
import java.net.URISyntaxException;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.Duration;

/**
 * Base class for Copilot Server API clients
 */
public class CopilotServerClient {

    protected static final String SERVER_URL_ENV = "copilot.serverBaseUrl";

    private final HttpClient httpClient;

    private final ObjectMapper objectMapper;

    /**
     * Constructor initializing HttpClient and ObjectMapper
     */
    public CopilotServerClient() {
        this.httpClient = HttpClient.newBuilder()
                .version(HttpClient.Version.HTTP_2)
                .followRedirects(HttpClient.Redirect.NORMAL).build();
        this.objectMapper = new ObjectMapper();
    }

    protected HttpClient getHttpClient() {
        return httpClient;
    }

    protected HttpRequest buildRequest(URI uri, String json) {
        return HttpRequest.newBuilder().uri(uri)
                .POST(HttpRequest.BodyPublishers.ofString(json))
                .header("Content-Type", "application/json")
                .timeout(Duration.ofSeconds(120)).build();
    }

    protected String writeAsJsonString(Object obj) {
        try {
            return objectMapper.writeValueAsString(obj);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException("Invalid data", e);
        }
    }

    protected <T> T readValue(String string, Class<T> clazz) {
        try {
            return objectMapper.readValue(string, clazz);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException("Invalid data", e);
        }
    }

    protected URI getQueryURI(String path) {
        try {
            return new URI(getServerBaseUrl() + path);
        } catch (URISyntaxException e) {
            throw new IllegalStateException(
                    "Invalid server configuration, server uri is wrong", e);
        }
    }

    protected String getServerBaseUrl() {
        String serverBaseUrl = System.getenv(SERVER_URL_ENV);
        if (serverBaseUrl != null) {
            return serverBaseUrl;
        }

        if (isDevelopment()) {
            // Try a localhost server
            try {
                String localhostUrl = "http://localhost:8081/v1/";
                String statusUrl = localhostUrl + "actuator/health";

                HttpRequest request = HttpRequest.newBuilder()
                        .uri(new URI(statusUrl)).timeout(Duration.ofSeconds(1))
                        .build();
                HttpResponse<String> response = httpClient.send(request,
                        HttpResponse.BodyHandlers.ofString());
                if (response.statusCode() == 200) {
                    return localhostUrl;
                }
            } catch (Exception e) {
                // Ignore
                getLogger().error(e.getMessage(), e);
                Thread.currentThread().interrupt();
            }
            return "https://copilot.stg.vaadin.com/v1/";
        } else {
            return "https://copilot.vaadin.com/v1/";
        }
    }

    protected static boolean isDevelopment() {
        return System.getProperty("copilot.development") != null;
    }

    protected Logger getLogger() {
        return LoggerFactory.getLogger(getClass());
    }

}
