/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.xwiki.rest.resources.pages;

import java.util.ArrayList;
import java.util.List;

import javax.ws.rs.GET;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.UriBuilder;
import javax.ws.rs.core.Response.Status;

import org.xwiki.component.annotation.Component;
import org.xwiki.rest.Relations;
import org.xwiki.rest.Utils;
import org.xwiki.rest.model.jaxb.Link;
import org.xwiki.rest.model.jaxb.Tag;
import org.xwiki.rest.model.jaxb.Tags;
import org.xwiki.rest.resources.tags.PagesForTagsResource;

import com.xpn.xwiki.XWikiException;
import com.xpn.xwiki.api.Document;
import com.xpn.xwiki.doc.XWikiDocument;
import com.xpn.xwiki.objects.BaseObject;
import com.xpn.xwiki.objects.BaseProperty;
import com.xpn.xwiki.objects.classes.BaseClass;

/**
 * @version $Id: PageTagsResource.java 25328 2009-11-27 17:53:57Z fmancinelli $
 */
@Component("org.xwiki.rest.resources.pages.PageTagsResource")
@Path("/wikis/{wikiName}/spaces/{spaceName}/pages/{pageName}/tags")
public class PageTagsResource extends ModifiablePageResource
{
    @GET
    public Tags getPageTags(@PathParam("wikiName") String wikiName, @PathParam("spaceName") String spaceName,
        @PathParam("pageName") String pageName) throws XWikiException
    {
        String pageId = Utils.getPageId(wikiName, spaceName, pageName);
        List<String> tagNames = getTagsFromDocument(pageId);

        Tags tags = objectFactory.createTags();
        for (String tagName : tagNames) {
            Tag tag = objectFactory.createTag();
            tag.setName(tagName);

            String tagUri =
                UriBuilder.fromUri(uriInfo.getBaseUri()).path(PagesForTagsResource.class).build(wikiName, tagName)
                    .toString();
            Link tagLink = objectFactory.createLink();
            tagLink.setHref(tagUri);
            tagLink.setRel(Relations.TAG);
            tag.getLinks().add(tagLink);

            tags.getTags().add(tag);
        }

        return tags;
    }

    @PUT
    public Response setTags(@PathParam("wikiName") String wikiName, @PathParam("spaceName") String spaceName,
        @PathParam("pageName") String pageName, Tags tags) throws XWikiException
    {
        DocumentInfo documentInfo = getDocumentInfo(wikiName, spaceName, pageName, null, null, true, false);

        Document doc = documentInfo.getDocument();

        if (!doc.hasAccessLevel("edit", Utils.getXWikiUser(componentManager))) {
            throw new WebApplicationException(Status.UNAUTHORIZED);
        }

        List<String> tagNames = new ArrayList<String>();
        for (Tag tag : tags.getTags()) {
            tagNames.add(tag.getName());
        }

        XWikiDocument xwikiDocument =
            Utils.getXWiki(componentManager).getDocument(doc.getPrefixedFullName(),
                Utils.getXWikiContext(componentManager));
        BaseObject xwikiObject = xwikiDocument.getObject("XWiki.TagClass", 0);

        if (xwikiObject == null) {
            int objectNumber = xwikiDocument.createNewObject("XWiki.TagClass", Utils.getXWikiContext(componentManager));
            xwikiObject = xwikiDocument.getObject("XWiki.TagClass", objectNumber);
            if (xwikiObject == null) {
                throw new WebApplicationException(Status.INTERNAL_SERVER_ERROR);
            }

            // We must initialize all the fields to an empty value in order to correctly create the object
            BaseClass xwikiClass =
                Utils.getXWiki(componentManager).getClass(xwikiObject.getClassName(),
                    Utils.getXWikiContext(componentManager));
            for (java.lang.Object propertyNameObject : xwikiClass.getPropertyNames()) {
                String propertyName = (String) propertyNameObject;
                xwikiObject.set(propertyName, "", Utils.getXWikiContext(componentManager));
            }
        }

        xwikiObject.set("tags", tagNames, Utils.getXWikiContext(componentManager));

        doc.save();

        return Response.status(Status.ACCEPTED).entity(tags).build();
    }

    private List<String> getTagsFromDocument(String documentId) throws XWikiException
    {
        XWikiDocument document =
            Utils.getXWiki(componentManager).getDocument(documentId, Utils.getXWikiContext(componentManager));
        BaseObject object = document.getObject("XWiki.TagClass");
        if (object != null) {
            BaseProperty prop = (BaseProperty) object.safeget("tags");
            if (prop != null) {
                return (List<String>) prop.getValue();
            }
        }

        return new ArrayList<String>();
    }
}
