/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package com.xpn.xwiki.wysiwyg.client.plugin.link.ui;

import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.ClickListener;
import com.google.gwt.user.client.ui.ClickListenerCollection;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.TextBox;
import com.xpn.xwiki.wysiwyg.client.editor.Strings;

/**
 * Abstract class to use as the superclass for the link generating tabs. Implements basic functionality for the link
 * tabs and stores basic data.
 * 
 * @version $Id$
 */
public abstract class AbstractHasLinkTab extends Composite implements HasLink
{
    /**
     * Click listeners for this click events source.
     */
    private final ClickListenerCollection clickListeners = new ClickListenerCollection();

    /**
     * The result link string.
     */
    private String link;

    /**
     * The HTML string for the label of the link to be created.
     */
    private String labelHTML;

    /**
     * The text for the label of the link to be created (stripped of potential HTML tags).
     */
    private String labelText;

    /**
     * The text box where the user will insert the text of the link to create (if any).
     */
    private TextBox labelTextBox = new TextBox();

    /**
     * @return the labelTextBox
     */
    protected TextBox getLabelTextBox()
    {
        return labelTextBox;
    }

    /**
     * @param labelTextBox the labelTextBox to set
     */
    protected void setLabelTextBox(TextBox labelTextBox)
    {
        this.labelTextBox = labelTextBox;
    }

    /**
     * Setter for the HTML link string generated by this widget.
     * 
     * @param link the link to set to this widget.
     */
    protected void setLink(String link)
    {
        this.link = link;
    }

    /**
     * @return the click listener collection of this click events source.
     */
    protected ClickListenerCollection getClickListeners()
    {
        return clickListeners;
    }

    /**
     * Setter for value of field <i>labelHTML</i>.
     * 
     * @param label the new value for <i>labelHTML</i>
     */
    protected void setLabelHTML(String label)
    {
        this.labelHTML = label;
    }

    /**
     * @return the HTML of the label of the link to be generated by this widget.
     */
    protected String getLabelHTML()
    {
        return labelHTML;
    }

    /**
     * @return the labelText
     */
    protected String getLabelText()
    {
        return labelText;
    }

    /**
     * @param labelText the labelText to set
     */
    protected void setLabelText(String labelText)
    {
        this.labelText = labelText;
        if (labelTextBox != null) {
            labelTextBox.setText(labelText);
        }
    }

    /**
     * {@inheritDoc}
     * 
     * @see HasLink#setLabel(String, String, boolean)
     */
    public void setLabel(String labelHTML, String labelText, boolean readOnly)
    {
        setLabelHTML(labelHTML);
        setLabelText(labelText);
        this.labelTextBox.setReadOnly(readOnly);
    }

    /**
     * {@inheritDoc}
     * 
     * @see HasLink#addClickListener(ClickListener)
     */
    public void addClickListener(ClickListener listener)
    {
        clickListeners.add(listener);
    }

    /**
     * {@inheritDoc}
     * 
     * @see HasLink#removeClickListener(ClickListener)
     */
    public void removeClickListener(ClickListener listener)
    {
        clickListeners.remove(listener);
    }

    /**
     * @return the url give by user.
     */
    public String getLink()
    {
        return link;
    }

    /**
     * Get the label for the link to generate. If the label text was modified, use that one, otherwise use the HTML
     * label.
     * 
     * @return the label for the link to generate.
     */
    protected String getLinkLabel()
    {
        if (this.labelTextBox.getText().trim().equals(this.getLabelText())) {
            return getLabelHTML();
        } else {
            return this.labelTextBox.getText().trim();
        }
    }

    /**
     * {@inheritDoc}
     */
    public boolean validateUserInput()
    {
        // the label cannot be void, for the moment
        // TODO: find a way to allow void labels to be set and get the automatically generated labels.
        if (this.labelTextBox.getText().trim().length() == 0) {
            Window.alert(Strings.INSTANCE.linkNoLabelError());
            return false;
        }
        return true;
    }
}
