package de.flapdoodle.embed.process.config.store;

import de.flapdoodle.embed.process.distribution.ArchiveType;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * Immutable implementation of {@link Package}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutablePackage.builder()}.
 * Use the static factory method to create immutable instances:
 * {@code ImmutablePackage.of()}.
 */
@SuppressWarnings({"all"})
public final class ImmutablePackage implements Package {
  private final ArchiveType archiveType;
  private final FileSet fileSet;
  private final String url;
  private final String hint;

  private ImmutablePackage(
      ArchiveType archiveType,
      FileSet fileSet,
      String url) {
    this.archiveType = Objects.requireNonNull(archiveType, "archiveType");
    this.fileSet = Objects.requireNonNull(fileSet, "fileSet");
    this.url = Objects.requireNonNull(url, "url");
    this.hint = null;
  }

  private ImmutablePackage(
      ArchiveType archiveType,
      FileSet fileSet,
      String url,
      String hint) {
    this.archiveType = archiveType;
    this.fileSet = fileSet;
    this.url = url;
    this.hint = hint;
  }

  /**
   * @return The value of the {@code archiveType} attribute
   */
  @Override
  public ArchiveType archiveType() {
    return archiveType;
  }

  /**
   * @return The value of the {@code fileSet} attribute
   */
  @Override
  public FileSet fileSet() {
    return fileSet;
  }

  /**
   * @return The value of the {@code url} attribute
   */
  @Override
  public String url() {
    return url;
  }

  /**
   * @return The value of the {@code hint} attribute
   */
  @Override
  public Optional<String> hint() {
    return Optional.ofNullable(hint);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Package#archiveType() archiveType} attribute.
   * A value equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for archiveType
   * @return A modified copy of the {@code this} object
   */
  public final ImmutablePackage withArchiveType(ArchiveType value) {
    ArchiveType newValue = Objects.requireNonNull(value, "archiveType");
    if (this.archiveType == newValue) return this;
    return new ImmutablePackage(newValue, this.fileSet, this.url, this.hint);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Package#fileSet() fileSet} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for fileSet
   * @return A modified copy of the {@code this} object
   */
  public final ImmutablePackage withFileSet(FileSet value) {
    if (this.fileSet == value) return this;
    FileSet newValue = Objects.requireNonNull(value, "fileSet");
    return new ImmutablePackage(this.archiveType, newValue, this.url, this.hint);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Package#url() url} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for url
   * @return A modified copy of the {@code this} object
   */
  public final ImmutablePackage withUrl(String value) {
    String newValue = Objects.requireNonNull(value, "url");
    if (this.url.equals(newValue)) return this;
    return new ImmutablePackage(this.archiveType, this.fileSet, newValue, this.hint);
  }

  /**
   * Copy the current immutable object by setting a <i>present</i> value for the optional {@link Package#hint() hint} attribute.
   * @param value The value for hint
   * @return A modified copy of {@code this} object
   */
  public final ImmutablePackage withHint(String value) {
    String newValue = Objects.requireNonNull(value, "hint");
    if (Objects.equals(this.hint, newValue)) return this;
    return new ImmutablePackage(this.archiveType, this.fileSet, this.url, newValue);
  }

  /**
   * Copy the current immutable object by setting an optional value for the {@link Package#hint() hint} attribute.
   * An equality check is used on inner nullable value to prevent copying of the same value by returning {@code this}.
   * @param optional A value for hint
   * @return A modified copy of {@code this} object
   */
  public final ImmutablePackage withHint(Optional<String> optional) {
    String value = optional.orElse(null);
    if (Objects.equals(this.hint, value)) return this;
    return new ImmutablePackage(this.archiveType, this.fileSet, this.url, value);
  }

  /**
   * This instance is equal to all instances of {@code ImmutablePackage} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutablePackage
        && equalTo(0, (ImmutablePackage) another);
  }

  private boolean equalTo(int synthetic, ImmutablePackage another) {
    return archiveType.equals(another.archiveType)
        && fileSet.equals(another.fileSet)
        && url.equals(another.url)
        && Objects.equals(hint, another.hint);
  }

  /**
   * Computes a hash code from attributes: {@code archiveType}, {@code fileSet}, {@code url}, {@code hint}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + archiveType.hashCode();
    h += (h << 5) + fileSet.hashCode();
    h += (h << 5) + url.hashCode();
    h += (h << 5) + Objects.hashCode(hint);
    return h;
  }

  /**
   * Prints the immutable value {@code Package} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    StringBuilder builder = new StringBuilder("Package{");
    builder.append("archiveType=").append(archiveType);
    builder.append(", ");
    builder.append("fileSet=").append(fileSet);
    builder.append(", ");
    builder.append("url=").append(url);
    if (hint != null) {
      builder.append(", ");
      builder.append("hint=").append(hint);
    }
    return builder.append("}").toString();
  }

  /**
   * Construct a new immutable {@code Package} instance.
   * @param archiveType The value for the {@code archiveType} attribute
   * @param fileSet The value for the {@code fileSet} attribute
   * @param url The value for the {@code url} attribute
   * @return An immutable Package instance
   */
  public static ImmutablePackage of(ArchiveType archiveType, FileSet fileSet, String url) {
    return new ImmutablePackage(archiveType, fileSet, url);
  }

  /**
   * Creates an immutable copy of a {@link Package} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable Package instance
   */
  public static ImmutablePackage copyOf(Package instance) {
    if (instance instanceof ImmutablePackage) {
      return (ImmutablePackage) instance;
    }
    return ImmutablePackage.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutablePackage ImmutablePackage}.
   * <pre>
   * ImmutablePackage.builder()
   *    .archiveType(de.flapdoodle.embed.process.distribution.ArchiveType) // required {@link Package#archiveType() archiveType}
   *    .fileSet(de.flapdoodle.embed.process.config.store.FileSet) // required {@link Package#fileSet() fileSet}
   *    .url(String) // required {@link Package#url() url}
   *    .hint(String) // optional {@link Package#hint() hint}
   *    .build();
   * </pre>
   * @return A new ImmutablePackage builder
   */
  public static ImmutablePackage.Builder builder() {
    return new ImmutablePackage.Builder();
  }

  /**
   * Builds instances of type {@link ImmutablePackage ImmutablePackage}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static final class Builder {
    private static final long INIT_BIT_ARCHIVE_TYPE = 0x1L;
    private static final long INIT_BIT_FILE_SET = 0x2L;
    private static final long INIT_BIT_URL = 0x4L;
    private long initBits = 0x7L;

    private ArchiveType archiveType;
    private FileSet fileSet;
    private String url;
    private String hint;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code Package} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(Package instance) {
      Objects.requireNonNull(instance, "instance");
      archiveType(instance.archiveType());
      fileSet(instance.fileSet());
      url(instance.url());
      Optional<String> hintOptional = instance.hint();
      if (hintOptional.isPresent()) {
        hint(hintOptional);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link Package#archiveType() archiveType} attribute.
     * @param archiveType The value for archiveType 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder archiveType(ArchiveType archiveType) {
      this.archiveType = Objects.requireNonNull(archiveType, "archiveType");
      initBits &= ~INIT_BIT_ARCHIVE_TYPE;
      return this;
    }

    /**
     * Initializes the value for the {@link Package#fileSet() fileSet} attribute.
     * @param fileSet The value for fileSet 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder fileSet(FileSet fileSet) {
      this.fileSet = Objects.requireNonNull(fileSet, "fileSet");
      initBits &= ~INIT_BIT_FILE_SET;
      return this;
    }

    /**
     * Initializes the value for the {@link Package#url() url} attribute.
     * @param url The value for url 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder url(String url) {
      this.url = Objects.requireNonNull(url, "url");
      initBits &= ~INIT_BIT_URL;
      return this;
    }

    /**
     * Initializes the optional value {@link Package#hint() hint} to hint.
     * @param hint The value for hint
     * @return {@code this} builder for chained invocation
     */
    public final Builder hint(String hint) {
      this.hint = Objects.requireNonNull(hint, "hint");
      return this;
    }

    /**
     * Initializes the optional value {@link Package#hint() hint} to hint.
     * @param hint The value for hint
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder hint(Optional<String> hint) {
      this.hint = hint.orElse(null);
      return this;
    }

    /**
     * Builds a new {@link ImmutablePackage ImmutablePackage}.
     * @return An immutable instance of Package
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutablePackage build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutablePackage(archiveType, fileSet, url, hint);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_ARCHIVE_TYPE) != 0) attributes.add("archiveType");
      if ((initBits & INIT_BIT_FILE_SET) != 0) attributes.add("fileSet");
      if ((initBits & INIT_BIT_URL) != 0) attributes.add("url");
      return "Cannot build Package, some of required attributes are not set " + attributes;
    }
  }
}
