/*
 * @(#)DummyEmitter.java
 *
 * Copyright Swiss Reinsurance Company, Mythenquai 50/60, CH 8022 Zurich. All rights reserved.
 */

package com.github.rjeschke.txtmark;

import java.util.HashMap;

/**
 * @author Vadim Karnigin
 */
public class CleaningEmitter implements Emitter {

    private final HashMap<String, LinkRef> linkRefs = new HashMap<String, LinkRef>();

    private Decorator decorator;

    public CleaningEmitter(Decorator decorator) {
        this.decorator = decorator;
    }

    public CleaningEmitter(Configuration configuration) {
        this.decorator = configuration.decorator;
    }

    @Override
    public void setUseExtensions(boolean useExtensions) {

    }

    @Override
    public void addLinkRef(String key, LinkRef linkRef) {

    }

    @Override
    public void emit(final StringBuilder out, final Block root) {
        switch (root.type) {
            case RULER:
                decorator.horizontalRuler(out);
                return;
            case NONE:
            case XML:
                break;
            case HEADLINE:
                decorator.openHeadline(out, root.hlDepth);
                break;
            case PARAGRAPH:
                decorator.openParagraph(out);
                break;
            case CODE:
            case FENCED_CODE:
                decorator.openCodeBlock(out);
                break;
            case BLOCKQUOTE:
                decorator.openBlockquote(out);
                break;
            case UNORDERED_LIST:
                decorator.openUnorderedList(out);
                break;
            case ORDERED_LIST:
                decorator.openOrderedList(out);
                break;
            case LIST_ITEM:
                decorator.openListItem(out);
                break;
        }

        if (root.hasLines()) {
            this.emitLines(out, root);
        } else {
            Block block = root.blocks;
            while (block != null) {
                this.emit(out, block);
                block = block.next;
            }
        }

        switch (root.type) {
            case RULER:
            case NONE:
            case XML:
                break;
            case HEADLINE:
                decorator.closeHeadline(out, root.hlDepth);
                break;
            case PARAGRAPH:
                decorator.closeParagraph(out);
                break;
            case CODE:
            case FENCED_CODE:
                decorator.closeCodeBlock(out);
                break;
            case BLOCKQUOTE:
                decorator.closeBlockquote(out);
                break;
            case UNORDERED_LIST:
                decorator.closeUnorderedList(out);
                break;
            case ORDERED_LIST:
                decorator.closeOrderedList(out);
                break;
            case LIST_ITEM:
                decorator.closeListItem(out);
                break;
        }
    }

    private void emitLines(final StringBuilder out, final Block block) {
        this.emitMarkedLines(out, block.lines);
    }

    private void emitMarkedLines(final StringBuilder out, final Line lines)
    {
        Line line = lines;
        StringBuilder in =  new StringBuilder();
        while (line != null)
        {
            if (!line.isEmpty)
            {
                in.append(line.value);
            }
            line = line.next;
        }
        this.recursiveEmitLine(out, in.toString(), 0, MarkToken.NONE);
    }

    private int recursiveEmitLine(final StringBuilder out, final String in, final int start, final MarkToken token)
    {
        int pos = start, a, b;
        final StringBuilder temp = new StringBuilder();
        while (pos < in.length())
        {
            final MarkToken mt = this.getToken(in, pos);
            if (token != MarkToken.NONE
                    && (mt == token || token == MarkToken.EM_STAR && mt == MarkToken.STRONG_STAR || token == MarkToken.EM_UNDERSCORE
                    && mt == MarkToken.STRONG_UNDERSCORE))
            {
                return pos;
            }

            switch (mt)
            {
                case IMAGE:
                case LINK:
                    temp.setLength(0);
                    b = checkLink(temp, in, pos, mt);
                    if (b > 0)
                    {
                        out.append(temp);
                        pos = b;
                    }
                    else
                    {
                        out.append(in.charAt(pos));
                    }
                    break;
                case EM_STAR:
                case EM_UNDERSCORE:
                    temp.setLength(0);
                    b = this.recursiveEmitLine(temp, in, pos + 1, mt);
                    if (b > 0)
                    {
                        decorator.openEmphasis(out);
                        out.append(temp);
                        decorator.closeEmphasis(out);
                        pos = b;
                    }
                    else
                    {
                        out.append(in.charAt(pos));
                    }
                    break;
                case STRONG_STAR:
                case STRONG_UNDERSCORE:
                    temp.setLength(0);
                    b = this.recursiveEmitLine(temp, in, pos + 2, mt);
                    if (b > 0)
                    {
                        decorator.openStrong(out);
                        out.append(temp);
                        decorator.closeStrong(out);
                        pos = b + 1;
                    }
                    else
                    {
                        out.append(in.charAt(pos));
                    }
                    break;
                case SUPER:
                    temp.setLength(0);
                    b = this.recursiveEmitLine(temp, in, pos + 1, mt);
                    if (b > 0)
                    {
                        decorator.openSuper(out);
                        out.append(temp);
                        decorator.closeSuper(out);
                        pos = b;
                    }
                    else
                    {
                        out.append(in.charAt(pos));
                    }
                    break;
                case CODE_SINGLE:
                case CODE_DOUBLE:
                    a = pos + (mt == MarkToken.CODE_DOUBLE ? 2 : 1);
                    b = findToken(in, a, mt);
                    if (b > 0)
                    {
                        pos = b + (mt == MarkToken.CODE_DOUBLE ? 1 : 0);
                        while (a < b && in.charAt(a) == ' ')
                        {
                            a++;
                        }
                        if (a < b)
                        {
                            while (in.charAt(b - 1) == ' ')
                            {
                                b--;
                            }
                            decorator.openCodeSpan(out);
                            Utils.appendCode(out, in, a, b);
                            decorator.closeCodeSpan(out);
                        }
                    }
                    else
                    {
                        out.append(in.charAt(pos));
                    }
                    break;
                case HTML:
                    temp.setLength(0);
                    b = checkHtml(temp, in, pos);
                    if (b > 0)
                    {
                        out.append(temp);
                        pos = b;
                    } else {
                        out.append(in.charAt(pos));
                    }
                    break;
                case ENTITY:
                    temp.setLength(0);
                    b = checkEntity(temp, in, pos);
                    if (b > 0)
                    {
                        out.append(temp);
                        pos = b;
                    }
                    else
                    {
                        out.append(in.charAt(pos));
                    }
                    break;
                case ESCAPE:
                    pos++;
                default:
                    out.append(in.charAt(pos));
                    break;
            }
            pos++;
        }
        return -1;
    }

    private MarkToken getToken(final String in, final int pos)
    {
        final char c0 = pos > 0 ? whitespaceToSpace(in.charAt(pos - 1)) : ' ';
        final char c = whitespaceToSpace(in.charAt(pos));
        final char c1 = pos + 1 < in.length() ? whitespaceToSpace(in.charAt(pos + 1)) : ' ';
        final char c2 = pos + 2 < in.length() ? whitespaceToSpace(in.charAt(pos + 2)) : ' ';

        switch (c)
        {
            case '*':
                if (c1 == '*')
                {
                    return c0 != ' ' || c2 != ' ' ? MarkToken.STRONG_STAR : MarkToken.EM_STAR;
                }
                return c0 != ' ' || c1 != ' ' ? MarkToken.EM_STAR : MarkToken.NONE;
            case '_':
                if (c1 == '_')
                {
                    return c0 != ' ' || c2 != ' ' ? MarkToken.STRONG_UNDERSCORE : MarkToken.EM_UNDERSCORE;
                }
                return c0 != ' ' || c1 != ' ' ? MarkToken.EM_UNDERSCORE : MarkToken.NONE;
            case '!':
                if (c1 == '[')
                {
                    return MarkToken.IMAGE;
                }
                return MarkToken.NONE;
            case '[':
                return MarkToken.LINK;
            case ']':
                return MarkToken.NONE;
            case '`':
                return c1 == '`' ? MarkToken.CODE_DOUBLE : MarkToken.CODE_SINGLE;
            case '\\':
                switch (c1)
                {
                    case '\\':
                    case '[':
                    case ']':
                    case '(':
                    case ')':
                    case '{':
                    case '}':
                    case '#':
                    case '"':
                    case '\'':
                    case '.':
                    case '>':
                    case '<':
                    case '*':
                    case '+':
                    case '-':
                    case '_':
                    case '!':
                    case '`':
                    case '~':
                    case '^':
                        return MarkToken.ESCAPE;
                    default:
                        return MarkToken.NONE;
                }
            case '<':
                return MarkToken.HTML;
            case '&':
                return MarkToken.ENTITY;
            default:
                return MarkToken.NONE;
        }
    }

    private int checkLink(final StringBuilder out, final String in, final int start, final MarkToken token)
    {
        boolean isAbbrev = false;
        int pos = start + (token == MarkToken.LINK ? 1 : 2);
        final StringBuilder temp = new StringBuilder();

        temp.setLength(0);
        pos = Utils.readMdLinkId(temp, in, pos);
        if (pos < start)
        {
            return -1;
        }

        final String name = temp.toString();
        String link = null, comment = null;
        final int oldPos = pos++;
        pos = Utils.skipSpaces(in, pos);
        if (pos < start)
        {
            final LinkRef lr = this.linkRefs.get(name.toLowerCase());
            if (lr != null)
            {
                isAbbrev = lr.isAbbrev;
                link = lr.link;
                comment = lr.title;
                pos = oldPos;
            }
            else
            {
                return -1;
            }
        }
        else if (in.charAt(pos) == '(')
        {
            pos++;
            pos = Utils.skipSpaces(in, pos);
            if (pos < start)
            {
                return -1;
            }
            temp.setLength(0);
            final boolean useLt = in.charAt(pos) == '<';
            pos = useLt ? Utils.readUntil(temp, in, pos + 1, '>') : Utils.readMdLink(temp, in, pos);
            if (pos < start)
            {
                return -1;
            }
            if (useLt)
            {
                pos++;
            }
            link = temp.toString();

            if (in.charAt(pos) == ' ')
            {
                pos = Utils.skipSpaces(in, pos);
                if (pos > start && in.charAt(pos) == '"')
                {
                    pos++;
                    temp.setLength(0);
                    pos = Utils.readUntil(temp, in, pos, '"');
                    if (pos < start)
                    {
                        return -1;
                    }
                    comment = temp.toString();
                    pos++;
                    pos = Utils.skipSpaces(in, pos);
                    if (pos == -1)
                    {
                        return -1;
                    }
                }
            }
            if (in.charAt(pos) != ')')
            {
                return -1;
            }
        }
        else if (in.charAt(pos) == '[')
        {
            pos++;
            temp.setLength(0);
            pos = Utils.readRawUntil(temp, in, pos, ']');
            if (pos < start)
            {
                return -1;
            }
            final String id = temp.length() > 0 ? temp.toString() : name;
            final LinkRef lr = this.linkRefs.get(id.toLowerCase());
            if (lr != null)
            {
                link = lr.link;
                comment = lr.title;
            }
        }
        else
        {
            final LinkRef lr = this.linkRefs.get(name.toLowerCase());
            if (lr != null)
            {
                isAbbrev = lr.isAbbrev;
                link = lr.link;
                comment = lr.title;
                pos = oldPos;
            }
            else
            {
                return -1;
            }
        }

        if (link == null)
        {
            return -1;
        }

        if (token == MarkToken.LINK)
        {
            if (isAbbrev && comment != null)
            {
                return -1;
            } else {
                decorator.openLink(out);
                Utils.appendValue(out, link, 0, link.length());
                if (comment != null)
                {
                    Utils.appendValue(out, comment, 0, comment.length());
                }
                this.recursiveEmitLine(out, name, 0, MarkToken.NONE);
                decorator.closeLink(out);
            }
        }
        else
        {
            decorator.openImage(out);
            Utils.appendValue(out, link, 0, link.length());
            Utils.appendValue(out, name, 0, name.length());
            if (comment != null)
            {
                Utils.appendValue(out, comment, 0, comment.length());
            }
            decorator.closeImage(out);
        }

        return pos;
    }

    private int findToken(final String in, final int start, final MarkToken token)
    {
        int pos = start;
        while (pos < in.length())
        {
            if (this.getToken(in, pos) == token)
            {
                return pos;
            }
            pos++;
        }
        return -1;
    }

    private int checkHtml(final StringBuilder out, final String in, final int start)
    {
        final StringBuilder temp = new StringBuilder();
        int pos;

        // Check for auto links
        temp.setLength(0);
        pos = Utils.readUntil(temp, in, start + 1, ':', ' ', '>', '\n');
        if (pos != -1 && in.charAt(pos) == ':' && HTML.isLinkPrefix(temp.toString()))
        {
            pos = Utils.readUntil(temp, in, pos, '>');
            if (pos != -1)
            {
                final String link = temp.toString();
                decorator.openLink(out);
                out.append(" href=\"");
                Utils.appendValue(out, link, 0, link.length());
                out.append("\">");
                Utils.appendValue(out, link, 0, link.length());
                decorator.closeLink(out);
                return pos;
            }
        }

        // Check for mailto auto link
        temp.setLength(0);
        pos = Utils.readUntil(temp, in, start + 1, '@', ' ', '>', '\n');
        if (pos != -1 && in.charAt(pos) == '@')
        {
            pos = Utils.readUntil(temp, in, pos, '>');
            if (pos != -1)
            {
                final String link = temp.toString();
                decorator.openLink(out);
                out.append(" href=\"");
                Utils.appendMailto(out, "mailto:", 0, 7);
                Utils.appendMailto(out, link, 0, link.length());
                out.append("\">");
                Utils.appendMailto(out, link, 0, link.length());
                decorator.closeLink(out);
                return pos;
            }
        }

        // Check for inline html
        if (start + 2 < in.length())
        {
            temp.setLength(0);
            return Utils.readXML(out, in, start, true);
        }

        return -1;
    }

    private static int checkEntity(final StringBuilder out, final String in, final int start)
    {
        final int pos = Utils.readUntil(out, in, start, ';');
        if (pos < 0 || out.length() < 3)
        {
            return -1;
        }
        if (out.charAt(1) == '#')
        {
            if (out.charAt(2) == 'x' || out.charAt(2) == 'X')
            {
                if (out.length() < 4)
                {
                    return -1;
                }
                for (int i = 3; i < out.length(); i++)
                {
                    final char c = out.charAt(i);
                    if ((c < '0' || c > '9') && ((c < 'a' || c > 'f') && (c < 'A' || c > 'F')))
                    {
                        return -1;
                    }
                }
            }
            else
            {
                for (int i = 2; i < out.length(); i++)
                {
                    final char c = out.charAt(i);
                    if (c < '0' || c > '9')
                    {
                        return -1;
                    }
                }
            }
            out.append(';');
        }
        else
        {
            for (int i = 1; i < out.length(); i++)
            {
                final char c = out.charAt(i);
                if (!Character.isLetterOrDigit(c))
                {
                    return -1;
                }
            }
            out.append(';');
            return HTML.isEntity(out.toString()) ? pos : -1;
        }

        return pos;
    }

    private static char whitespaceToSpace(final char c)
    {
        return Character.isWhitespace(c) ? ' ' : c;
    }

}
