/*
	Copyright 2009 Anatol Gregory Mayen
	
	Licensed under the Apache License, Version 2.0 (the "License");
	you may not use this file except in compliance with the License. 
	You may obtain a copy of the License at 
	
	http://www.apache.org/licenses/LICENSE-2.0 
	
	Unless required by applicable law or agreed to in writing, software 
	distributed under the License is distributed on an "AS IS" BASIS, 
	WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
	See the License for the specific language governing permissions and 
	limitations under the License. 
*/
package eu.maydu.gwt.validation.client.actions;

import java.util.LinkedList;
import java.util.List;

import com.google.gwt.user.client.ui.TextBox;
import com.google.gwt.user.client.ui.UIObject;

import eu.maydu.gwt.validation.client.ValidationAction;
import eu.maydu.gwt.validation.client.ValidationResult;


/**
 * 
 * This action is used to change the styles of visual components
 * in case of a failed validation.
 * 
 * @author Anatol Gregory Mayen
 *
 */
public class StyleAction extends ValidationAction<UIObject>{

	private List<UIObject> objects = new LinkedList<UIObject>();
	private List<UIObject> customObjects = new LinkedList<UIObject>();
	private String[] stylenameAdd, stylenameRemove;
	private boolean isDependentStyle = false;
	private boolean firstRun = true;
	
	public StyleAction(String[] stylenameAdd, boolean isDependentStyle) {
		super();
		this.stylenameAdd = stylenameAdd;
		this.isDependentStyle = isDependentStyle;
	}
	
	public StyleAction(String[] stylenameAdd, UIObject customObject, boolean isDependentStyle) {
		super();
		this.stylenameAdd = stylenameAdd;
		this.isDependentStyle = isDependentStyle;
		this.customObjects.add(customObject);
	}
	
	public StyleAction(String[] stylenameAdd) {
		super();
		this.stylenameAdd = stylenameAdd;
	}
	
	public StyleAction(String[] stylenameAdd, UIObject customObject) {
		super();
		this.stylenameAdd = stylenameAdd;
		this.customObjects.add(customObject);
	}
	
	public StyleAction(String add, String remove, boolean isDependentStyle) {
		this(add, remove, null, isDependentStyle);
	}
	
	public StyleAction(String add, String remove, UIObject customObject, boolean isDependentStyle) {
		super();
		this.stylenameAdd = new String[1];
		this.stylenameAdd[0] = add;
		
		this.stylenameRemove = new String[1];
		this.stylenameRemove[0] = remove;
		this.isDependentStyle = isDependentStyle;
		this.customObjects.add(customObject);
	}
	
	public StyleAction(String add, String remove) {
		this(add, remove, null);
	}
	
	public StyleAction(String add, String remove, UIObject customObject) {
		super();
		this.stylenameAdd = new String[1];
		this.stylenameAdd[0] = add;
		
		this.stylenameRemove = new String[1];
		this.stylenameRemove[0] = remove;
		this.customObjects.add(customObject);
	}
	
	public StyleAction(String add, boolean isDependentStyle) {
		super();
		this.stylenameAdd = new String[1];
		this.stylenameAdd[0] = add;
		this.isDependentStyle = isDependentStyle;
	}
	
	public StyleAction(String add, UIObject customObject, boolean isDependentStyle) {
		super();
		this.stylenameAdd = new String[1];
		this.stylenameAdd[0] = add;
		this.stylenameRemove = null;
		this.isDependentStyle = isDependentStyle;
		this.customObjects.add(customObject);
	}
	
	public StyleAction(String add) {
		super();
		this.stylenameAdd = new String[1];
		this.stylenameAdd[0] = add;
	}
	
	public StyleAction(String add, UIObject customObject) {
		super();
		this.stylenameAdd = new String[1];
		this.stylenameAdd[0] = add;
		this.stylenameRemove = null;
		this.customObjects.add(customObject);
	}
		
	public StyleAction(String[] stylenameAdd, String stylenameRemove[], boolean isDependentStyle) {
		super();
		this.stylenameAdd = stylenameAdd;
		this.stylenameRemove = stylenameRemove;
		this.isDependentStyle = isDependentStyle;
	}
	
	public StyleAction(String[] stylenameAdd, String stylenameRemove[], UIObject customObject, boolean isDependentStyle) {
		super();
		this.stylenameAdd = stylenameAdd;
		this.stylenameRemove = stylenameRemove;
		this.isDependentStyle = isDependentStyle;
		this.customObjects.add(customObject);
	}
	
	public StyleAction(String[] stylenameAdd, String stylenameRemove[]) {
		super();
		this.stylenameAdd = stylenameAdd;
		this.stylenameRemove = stylenameRemove;
	}
	
	public StyleAction(String[] stylenameAdd, String stylenameRemove[], UIObject customObject) {
		super();
		this.stylenameAdd = stylenameAdd;
		this.stylenameRemove = stylenameRemove;
		this.customObjects.add(customObject);
	}
	

	


	
	@Override
	public void invoke(ValidationResult result, UIObject object) {
		firstRun = false;
		
		if(this.customObjects.isEmpty() && object == null)
			throw new IllegalArgumentException("An non-null UIObject is needed!");
		if(!this.objects.contains(object))
			this.objects.add(object);
		removeStyles(this.stylenameRemove, null);
		addStyles(this.stylenameAdd, null);
	}
		
	/**
	 * Resets all the objects that were "registered" to this
	 * <code>StyleAction</code>. Registered are all the <code>UIObjects</code>
	 * that were given by an invocation of 
	 * {@link #invoke(ValidationResult, UIObject)}.
	 * 
	 */
	public void reset() {
		if(firstRun) //take care that invoke was called previously
			return;
		removeStyles(this.stylenameAdd, null);
		addStyles(this.stylenameRemove, null);
	}
	
	
	/**
	 * Resets only the given object.
	 */
	public void reset(UIObject o) {
		if(firstRun) //take care that invoke was called previously
			return;
		removeStyles(this.stylenameAdd, o);
		addStyles(this.stylenameRemove, o);
	}
	
	
	/**
	 * Removes all the styles from all the "registered" <code>UIObjects</code>.
	 * 
	 * If <code>target</code> is <em>non-null</em> only the <em>target</em>
	 * <code>UIObject</code> is affected. This allows for a more fine grained
	 * behaviour.
	 * 
	 * @param styles The styles to remove
	 * @param target If non-null all styles are only removed from the given UIObject
	 */
	private void removeStyles(String[] styles, UIObject target) {
		if(styles == null)
			return;
		for(String s : styles) {
			if(s != null && s.length() > 0) {
				if(this.isDependentStyle) {
					if(target != null) {
						target.removeStyleDependentName(s);
						continue;
					}
					if(this.customObjects.isEmpty()) {
						for(UIObject o : this.objects)
							o.removeStyleDependentName(s);
					}else {
						for(UIObject o : this.customObjects)
							o.removeStyleDependentName(s);
					}
				}else {
					if(target != null) {
						target.removeStyleName(s);
						continue;
					}
					if(this.customObjects.isEmpty()) {
						for(UIObject o : this.objects) {
							o.removeStyleName(s);
						}
					}else {
						for(UIObject o : this.customObjects) {
							o.removeStyleName(s);
						}
					}
				}
			}
		}
	}
		
	
	/**
	 * Adds all the styles to all the "registered" <code>UIObjects</code>.
	 * 
	 * If <code>target</code> is <em>non-null</em> only the <em>target</em>
	 * <code>UIObject</code> is affected. This allows for a more fine grained
	 * behaviour.
	 * 
	 * @param styles The styles to add
	 * @param target If non-null all styles are only added to the given UIObject
	 */
	private void addStyles(String[] styles, UIObject target) {
		if(styles == null)
			return;
		for(String s : styles) {
			if(this.isDependentStyle) {
				if(target != null) {
					target.addStyleDependentName(s);
					continue;
				}
				if(this.customObjects.isEmpty()) {
					for(UIObject o : this.objects)
						o.addStyleDependentName(s);
				}else {
					for(UIObject o : this.customObjects)
						o.addStyleDependentName(s);
				}
			}else {
				if(target != null) {
					target.addStyleName(s);
					continue;
				}
				if(this.customObjects.isEmpty()) {
					for(UIObject o : this.objects) {
						o.addStyleName(s);
					}
				}else {
					for(UIObject o : this.customObjects) {
						o.addStyleName(s);
					}
				}
			}
				
		}
	}
}