/*
	Copyright 2009 Anatol Gregory Mayen
	
	Licensed under the Apache License, Version 2.0 (the "License");
	you may not use this file except in compliance with the License. 
	You may obtain a copy of the License at 
	
	http://www.apache.org/licenses/LICENSE-2.0 
	
	Unless required by applicable law or agreed to in writing, software 
	distributed under the License is distributed on an "AS IS" BASIS, 
	WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
	See the License for the specific language governing permissions and 
	limitations under the License. 
*/
package eu.maydu.gwt.validation.client.validators.multifield;

import com.google.gwt.user.client.ui.SuggestBox;
import com.google.gwt.user.client.ui.TextBoxBase;

import eu.maydu.gwt.validation.client.MultiFieldValidator;
import eu.maydu.gwt.validation.client.ValidationAction;
import eu.maydu.gwt.validation.client.ValidationResult;
import eu.maydu.gwt.validation.client.i18n.ValidationMessages;

/**
 * The <code>MultiStringLengthValidator</code> compares the data of
 * multiple given input fields. If their input length does not lie inside the 
 * specified interval a <code>ValidationError</code> is issued.
 * 
 * Useful for password matching, etc.
 * 
 * @author Anatol Mayen
 *
 */
public class MultiStringLengthValidator extends MultiFieldValidator<MultiStringLengthValidator> {

	private boolean trim = false;
	private int min, max;
	
	public MultiStringLengthValidator(boolean preventsPropagationOfValidationChain, boolean trim, int min, int max, TextBoxBase... base) {
		this(preventsPropagationOfValidationChain, trim, min, max, null, base);
	}
	
	public MultiStringLengthValidator(boolean preventsPropagationOfValidationChain, boolean trim, int min, int max, String customMsgKey, TextBoxBase... base) {
		super(preventsPropagationOfValidationChain, base);
		if(min < 0 || max < min)
			throw new IllegalArgumentException("Min - Max parameters not correct");
		this.trim = trim;
		this.min = min;
		this.max = max;
		this.setCustomMsgKey(customMsgKey);
	}
	
	public MultiStringLengthValidator(boolean preventsPropagationOfValidationChain, boolean trim, int min, int max, SuggestBox... base) {
		this(preventsPropagationOfValidationChain, trim, min, max, null, base);
	}
	
	public MultiStringLengthValidator(boolean preventsPropagationOfValidationChain, boolean trim, int min, int max, String customMsgKey, SuggestBox... base) {
		super(preventsPropagationOfValidationChain, base);
		if(min < 0 || max < min)
			throw new IllegalArgumentException("Min - Max parameters not correct");
		this.trim = trim;
		this.min = min;
		this.max = max;
		this.setCustomMsgKey(customMsgKey);
	}
	
	public MultiStringLengthValidator(boolean preventsPropagationOfValidationChain, int min, int max, TextBoxBase... base) {
		this(preventsPropagationOfValidationChain, false, min, max, base);
	}
	
	public MultiStringLengthValidator(boolean preventsPropagationOfValidationChain, int min, int max, String customMsgKey, TextBoxBase... base) {
		this(preventsPropagationOfValidationChain, false, min, max, customMsgKey, base);
	}
	
	public MultiStringLengthValidator(boolean preventsPropagationOfValidationChain, int min, int max, SuggestBox... base) {
		this(preventsPropagationOfValidationChain, false, min, max, base);
	}
	
	public MultiStringLengthValidator(boolean preventsPropagationOfValidationChain, int min, int max, String customMsgKey, SuggestBox... base) {
		this(preventsPropagationOfValidationChain, false, min, max, customMsgKey, base);
	}
	
	public MultiStringLengthValidator(int min, int max, boolean trim, TextBoxBase... base) {
		this(false, trim, min, max, base);
	}
	
	public MultiStringLengthValidator(int min, int max, boolean trim, String customMsgKey, TextBoxBase... base) {
		this(false, trim, min, max, customMsgKey, base);
	}
	
	public MultiStringLengthValidator(int min, int max, boolean trim, SuggestBox... base) {
		this(false, trim, min, max, base);
	}
	
	public MultiStringLengthValidator(int min, int max, boolean trim, String customMsgKey, SuggestBox... base) {
		this(false, trim, min, max, customMsgKey, base);
	}
	
	public MultiStringLengthValidator(int min, int max, TextBoxBase... base) {
		this(false, false, min, max, base);
	}
	
	public MultiStringLengthValidator(int min, int max, String customMsgKey, TextBoxBase... base) {
		this(false, false, min, max, customMsgKey, base);
	}
	
	public MultiStringLengthValidator(int min, int max, SuggestBox... base) {
		this(false, false, min, max, base);
	}
	
	public MultiStringLengthValidator(int min, int max, String customMsgKey, SuggestBox... base) {
		this(false, false, min, max, customMsgKey, base);
	}
	
	
	
	@Override
	public void invokeActions(ValidationResult result) {
		for(ValidationAction va : getFailureActions()) {
			
			//Invoke all the actions for all the input fields
			
			if(this.isInTextBoxMode()) {
				for(TextBoxBase base : getTextBoxes())
					va.invoke(result, base);
			}else {
				for(SuggestBox base : getSuggestBoxes())
					va.invoke(result, base);
			}
		}
		
	}
	
	@Override
	public void resetActions() {
		for(ValidationAction va : getFailureActions()) {
			
			//Invoke all the actions for all the input fields
			
			if(this.isInTextBoxMode()) {
				for(TextBoxBase base : getTextBoxes())
					va.reset(base);
			}else {
				for(SuggestBox base : getSuggestBoxes())
					va.reset(base);
			}
		}
		
	}

	@Override
	public <V extends ValidationMessages> ValidationResult validate(V messages) {
	
		

		for(String s : getInputValues()) {
			
			if(trim)
				s = s.trim();
			
			if((s.length() < min || s.length() > max) && !(s.equals("") && !isRequired()))
				return new ValidationResult(getErrorMessage(messages, messages.getStandardMessages().length(this.min, this.max, s.length()),this.min, this.max, s.length()));
						
		}
		return null;
		
	}

	
	
}
