/*
	Copyright 2009 Anatol Gregory Mayen
	
	Licensed under the Apache License, Version 2.0 (the "License");
	you may not use this file except in compliance with the License. 
	You may obtain a copy of the License at 
	
	http://www.apache.org/licenses/LICENSE-2.0 
	
	Unless required by applicable law or agreed to in writing, software 
	distributed under the License is distributed on an "AS IS" BASIS, 
	WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
	See the License for the specific language governing permissions and 
	limitations under the License. 
*/
package eu.maydu.gwt.validation.client.validators.multifield;

import java.util.List;

import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.SuggestBox;
import com.google.gwt.user.client.ui.TextBoxBase;

import eu.maydu.gwt.validation.client.MultiFieldValidator;
import eu.maydu.gwt.validation.client.ValidationAction;
import eu.maydu.gwt.validation.client.ValidationResult;
import eu.maydu.gwt.validation.client.i18n.ValidationMessages;

/**
 * The <code>MultiStringsEqualsValidator</code> compares the data of
 * multiple given input fields. If they don't match an <code>ValidationError</code>
 * is issued.
 * 
 * Useful for password matching, etc.
 * 
 * @author Anatol Mayen
 *
 */
public class MultiStringsEqualsValidator extends MultiFieldValidator<MultiStringsEqualsValidator> {

	private boolean trim = false;
	private boolean ignoreCase = false;
	
	public MultiStringsEqualsValidator(boolean preventsPropagationOfValidationChain, boolean trim, boolean ignoreCase, TextBoxBase... base) {
		this(preventsPropagationOfValidationChain, trim, ignoreCase, null, base);
	}
	
	public MultiStringsEqualsValidator(boolean preventsPropagationOfValidationChain, boolean trim, boolean ignoreCase, String customMsgKey, TextBoxBase... base) {
		super(preventsPropagationOfValidationChain, base);
		this.trim = trim;
		this.ignoreCase = ignoreCase;
		this.setCustomMsgKey(customMsgKey);
	}
	
	public MultiStringsEqualsValidator(boolean preventsPropagationOfValidationChain, boolean trim, boolean ignoreCase, SuggestBox... base) {
		this(preventsPropagationOfValidationChain, trim, ignoreCase, null, base);
	}
	
	public MultiStringsEqualsValidator(boolean preventsPropagationOfValidationChain, boolean trim, boolean ignoreCase, String customMsgKey, SuggestBox... base) {
		super(preventsPropagationOfValidationChain, base);
		this.trim = trim;
		this.ignoreCase = ignoreCase;
		this.setCustomMsgKey(customMsgKey);
	}
	
	public MultiStringsEqualsValidator(boolean preventsPropagationOfValidationChain, TextBoxBase... base) {
		this(preventsPropagationOfValidationChain, false, false, base);
	}
	
	public MultiStringsEqualsValidator(boolean preventsPropagationOfValidationChain, String customMsgKey, TextBoxBase... base) {
		this(preventsPropagationOfValidationChain, false, false, customMsgKey, base);
	}
	
	public MultiStringsEqualsValidator(boolean preventsPropagationOfValidationChain, SuggestBox... base) {
		this(preventsPropagationOfValidationChain, false, false, base);
	}
	
	public MultiStringsEqualsValidator(boolean preventsPropagationOfValidationChain, String customMsgKey, SuggestBox... base) {
		this(preventsPropagationOfValidationChain, false, false, customMsgKey, base);
	}
	
	public MultiStringsEqualsValidator(boolean trim, boolean ignoreCase, TextBoxBase... base) {
		this(false, trim, ignoreCase, base);
	}
	
	public MultiStringsEqualsValidator(boolean trim, boolean ignoreCase, String customMsgKey, TextBoxBase... base) {
		this(false, trim, ignoreCase, customMsgKey, base);
	}
	
	public MultiStringsEqualsValidator(boolean trim, boolean ignoreCase, SuggestBox... base) {
		this(false, trim, ignoreCase, base);
	}
	
	public MultiStringsEqualsValidator(boolean trim, boolean ignoreCase, String customMsgKey, SuggestBox... base) {
		this(false, trim, ignoreCase, customMsgKey, base);
	}

	
	public MultiStringsEqualsValidator(TextBoxBase... base) {
		this(false, false, base);
	}
	
	public MultiStringsEqualsValidator(String customMsgKey, TextBoxBase... base) {
		this(false, false, customMsgKey, base);
	}
	
	public MultiStringsEqualsValidator(SuggestBox... base) {
		this(false, false, base);
	}
	
	public MultiStringsEqualsValidator(String customMsgKey, SuggestBox... base) {
		this(false, false, customMsgKey, base);
	}
	
	@Override
	public void invokeActions(ValidationResult result) {
		for(ValidationAction va : getFailureActions()) {
			
			//Invoke all the actions for all the input fields
			
			if(this.isInTextBoxMode()) {
				for(TextBoxBase base : getTextBoxes())
					va.invoke(result, base);
			}else {
				for(SuggestBox base : getSuggestBoxes())
					va.invoke(result, base);
			}
		}
		
	}
	
	@Override
	public void resetActions() {
		for(ValidationAction va : getFailureActions()) {
			
			//Invoke all the actions for all the input fields
			
			if(this.isInTextBoxMode()) {
				for(TextBoxBase base : getTextBoxes())
					va.reset(base);
			}else {
				for(SuggestBox base : getSuggestBoxes())
					va.reset(base);
			}
		}
		
	}

	@Override
	public <V extends ValidationMessages> ValidationResult validate(V messages) {
	
		
		List<String> values = this.getInputValues();
		String referenceValue = values.get(0);
		if(referenceValue.equals("") && isRequired())
			return new ValidationResult(getErrorMessage(messages, messages.getStandardMessages().notNull()));
		if(trim)
			referenceValue = referenceValue.trim();
		
		for(String s : values) {
			
			//case sensitive
			if(!ignoreCase && !s.equals(referenceValue))
				return new ValidationResult(getErrorMessage(messages, messages.getStandardMessages().equal()));
			
			//case insensitive
			if(ignoreCase && !s.equalsIgnoreCase(referenceValue))
				return new ValidationResult(getErrorMessage(messages, messages.getStandardMessages().equal()));
		}
		return null;
		
	}

	
	
}
