/*

 Copyright (c) 2005-2011, Carlos Amengual.

 Licensed under a BSD-style License. You can find the license here:
 http://www.informatica.info/projects/css/LICENSE.txt

 */

package info.informatica.doc.agent;


import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import org.apache.log4j.Logger;

/**
 * Asynchronous downloading of an embedded resource of a given type.
 * 
 * @author Carlos Amengual
 *
 */
abstract public class ResourceDownloader<C> extends Thread {
	
	private URL url;
	protected C nativeContent = null;
	protected List<DownloadListener<C>> listeners = new LinkedList<DownloadListener<C>>();
	private boolean done = false;
	
	static Logger log = Logger.getLogger(ResourceDownloader.class.getName());

	public ResourceDownloader(URL url) {
		this.url = url;
	}

	public URL getURL() {
		return url;
	}

	public void addImageListener(DownloadListener<C> listener) {
		if(done) {
			if(nativeContent == null) {
				listener.doFailedDownload();
			} else {
				listener.doContentDownloaded(nativeContent);	
			}
		} else {
			this.listeners.add(listener);
		}
	}

	public boolean isDone() {
		return done;
	}

	@Override
	public void run() {
		try {
			InputStream is = openStream();
			readContent(is);
			is.close();
			done = true;
			notifyDownload();
		}catch(IOException e) {
			notifyFailure(e);
		}
		listeners.clear();
	}

	protected InputStream openStream() throws IOException {
		return getURL().openStream();
	}

	/**
	 * Reads content from a stream and transforms it to native content.
	 * 
	 * @param is the stream with the content.
	 * @throws IOException if an error occurs when reading the stream.
	 */
	abstract protected void readContent(InputStream is) throws IOException;

	protected void notifyDownload() {
		Iterator<DownloadListener<C>> it = listeners.iterator();
		while(it.hasNext()) {
			it.next().doContentDownloaded(nativeContent);
		}
		nativeContent = null;
	}

	protected void notifyFailure(IOException e) {
		Iterator<DownloadListener<C>> it = listeners.iterator();
		while(it.hasNext()) {
			it.next().doFailedDownload();
		}
		log.warn(e);
	}

	public C getNativeContent() {
		return nativeContent;
	}

}
