/*

 Copyright (c) 2005-2011, Carlos Amengual.

 Licensed under a BSD-style License. You can find the license here:
 http://www.informatica.info/projects/css/LICENSE.txt

 */

package info.informatica.doc.dom4j;

import info.informatica.doc.agent.DownloadListener;
import info.informatica.doc.agent.ResourceDownloader;
import info.informatica.doc.agent.UserAgent;
import info.informatica.doc.style.css.StyleDatabase;
import info.informatica.doc.style.css.visual.ElementReplacer;
import info.informatica.doc.xml.dtd.DefaultEntityResolver;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PushbackReader;
import java.net.URL;
import java.net.URLConnection;
import java.util.HashMap;
import java.util.Map;

import org.dom4j.DocumentException;
import org.dom4j.io.SAXReader;
import org.xml.sax.EntityResolver;
import org.xml.sax.InputSource;

/**
 * User Agent based on DOM4J document trees.
 * <p>
 * 
 * @author Carlos Amengual (amengual at informatica.info)
 */
abstract public class DOM4JUserAgent<C> implements UserAgent<C> {
	
	private Map<URL,ResourceDownloader<C>> imageDownloaders = 
		new HashMap<URL,ResourceDownloader<C>>();
	
	private Map<String,ElementReplacer<C>> replacers = 
		new HashMap<String,ElementReplacer<C>>(4);

	protected DOM4JUserAgent() {
		super();
	}

	/**
	 * Convenience method that reads and parses an XHTML 
	 * document located in the given URL, with a default entity resolver.
	 * 
	 * @param url the URL that points to the document.
	 * @return the XHTMLDocument.
	 * @throws IOException
	 * @throws DocumentException
	 */
	public XHTMLDocument readURL(URL url)
	throws IOException, DocumentException {
		return readURL(url, new DefaultEntityResolver());
	}

	/**
	 * Reads and parses an XHTML document located in the given URL.
	 * 
	 * @param url the URL that points to the document.
	 * @param resolver the entity resolver.
	 * @return the XHTMLDocument.
	 * @throws IOException
	 * @throws DocumentException
	 */
	public XHTMLDocument readURL(URL url, EntityResolver resolver)
	throws IOException, DocumentException {
		URLConnection con = url.openConnection();
		String conType = con.getContentType();
		InputStream is = null;
		XHTMLDocument xdoc = null;
		try {
			is = con.getInputStream();
			String charset = null;
			if(conType != null) {
				int idx = conType.indexOf("charset=");
				if(idx>=0){
					charset = conType.substring(idx+8);
				}
			} else {
				conType = URLConnection.guessContentTypeFromStream(is);
			}
			// Handle UTF-8 BOM
			PushbackReader re;
			if(charset != null) {
				re = new PushbackReader(new InputStreamReader(is, charset), 1);
			} else {
				re = new PushbackReader(new InputStreamReader(is), 1);
			}
			int iread = re.read();
			if(iread != -1 && iread == 0xefbbbf) {
				re.unread(iread);
			} else {
				charset = "utf-8";
			}
			InputSource isrc = new InputSource(re);
			XHTMLDocumentFactory factory = (XHTMLDocumentFactory) 
				XHTMLDocumentFactory.getInstance();
			factory.setStyleDatabase(getStyleDatabase());
			SAXReader reader = new SAXReader(factory);
			reader.setEntityResolver(resolver);
			xdoc = (XHTMLDocument) reader.read(isrc);
			xdoc.setBaseURL(url);
		}catch(IOException e) {
			throw e;
		} finally {
			if(is != null) {
				is.close();
			}
		}
		return xdoc;
	}

	public ResourceDownloader<C> download(URL url) {
		ResourceDownloader<C> imgDwn;
		synchronized(imageDownloaders) {
			imgDwn = imageDownloaders.get(url);
			if(imgDwn == null) {
				imgDwn = createDownloader(url);
				imgDwn.start();
				imageDownloaders.put(url, imgDwn);
			}
		}
		return imgDwn;
	}

	public void addDownloadListener(URL url, DownloadListener<C> listener) {
		ResourceDownloader<C> imgDwn = download(url);
		synchronized(imgDwn) {
			imgDwn.addImageListener(listener);
		}
	}

	public ResourceDownloader<C> getResourceDownloader(URL url) {
		return imageDownloaders.get(url);
	}

	public ElementReplacer<C> getElementReplacer(String namespaceUri) {
		return replacers.get(namespaceUri);
	}

	public void setElementReplacer(String namespaceUri, 
			ElementReplacer<C> replacer) {
		synchronized(replacers) {
			replacers.put(namespaceUri, replacer);
			if(ElementReplacer.NAMESPACE_XHTML.equals(namespaceUri)) {
				replacers.put("", replacer);
			}
		}
	}

	abstract public StyleDatabase getStyleDatabase();
	
	abstract protected ResourceDownloader<C> createDownloader(URL url);

}