/*

 Copyright (c) 2005-2011, Carlos Amengual.

 Licensed under a BSD-style License. You can find the license here:
 http://www.informatica.info/projects/css/LICENSE.txt

 */

package info.informatica.doc.dom4j;

import info.informatica.doc.style.css.StyleDatabase;
import info.informatica.doc.style.css.dom.BaseCSSRule;
import info.informatica.doc.style.css.dom.BaseCSSStyleDeclaration;

import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.Set;

import org.apache.log4j.Logger;
import org.dom4j.Element;
import org.dom4j.dom.DOMDocument;
import org.dom4j.dom.DOMDocumentType;
import org.dom4j.dom.DOMElement;
import org.w3c.css.sac.CSSException;
import org.w3c.css.sac.InputSource;
import org.w3c.dom.DOMException;

/**
 * XHTML-specific implementation of a DOM4J <code>Document</code>.
 * 
 * @author Carlos Amengual
 * 
 */
public class XHTMLDocument extends DOMDocument {

	private static final long serialVersionUID = 3L;

	private DOM4JCSSStyleSheet defStyleSheet = null;
	
	private BaseCSSStyleDeclaration userImportantStyle = null;
	
	private DOM4JCSSStyleSheet mergedStyleSheet = null;
	
	private int styleCacheSerial = Integer.MIN_VALUE;
	
	private StyleDatabase styleDb = null;

	private URL baseURL = null;
	
	Set<LinkElement> linkedStyle = new LinkedHashSet<LinkElement>(4);

	Set<StyleElement> embeddedStyle = new LinkedHashSet<StyleElement>(3);

	static Logger log = Logger.getLogger(XHTMLDocument.class.getName());

	public XHTMLDocument() {
		super();
	}

	public XHTMLDocument(String name) {
		super(name);
	}

	public XHTMLDocument(DOMElement rootElement) {
		super(rootElement);
	}

	public XHTMLDocument(DOMDocumentType docType) {
		super(docType);
	}

	public XHTMLDocument(DOMElement rootElement, DOMDocumentType docType) {
		super(rootElement, docType);
	}

	public XHTMLDocument(String name, DOMElement rootElement, DOMDocumentType docType) {
		super(name, rootElement, docType);
	}

	@Override
    protected String elementID(Element element) {
        return element.attributeValue("id");
    }

	/**
	 * Gets the style sheet that applies to this document.
	 * <p>
	 * The style sheet is built lazily, starting with a default style sheet,
	 * and then adding the linked and embedded sheets in the document.
	 * 
	 * @return the style sheet that applies to this document.
	 */
	public DOM4JCSSStyleSheet getStyleSheet() {
		if(mergedStyleSheet == null){
			mergeStyleSheets();
		}
		return mergedStyleSheet;
	}

	private void mergeStyleSheets() {
		mergedStyleSheet = defStyleSheet.clone();
		/*
		 * Now we add the linked and embedded styles. Must be added 
		 * in this order, as mandated by the CSS spec.
		 */
		// Add styles referenced by links
		Iterator<LinkElement> links = linkedStyle.iterator();
		while(links.hasNext()){
			try {
				links.next().mergeStyle();
			} catch(CSSException e){
				log.error("Error merging linked style", e);
				continue;
			}
		}
		// Add embedded styles
		Iterator<StyleElement> embd = embeddedStyle.iterator();
		while(embd.hasNext()){
			try {
				embd.next().mergeStyle();
			} catch(CSSException e){
				log.error("Error merging embedded style", e);
				continue;
			}
		}
		if(userImportantStyle != null) {
			mergedStyleSheet.addRule(
					(BaseCSSRule) userImportantStyle.getParentRule());
		}
	}

	/**
	 * Adds a style sheet (contained by the given InputSource) to the 
	 * global style sheet defined by the document's default style sheet 
	 * and all the linked and embedded styles.
	 * 
	 * @param cssSrc
	 * @throws DOMException
	 * @throws IOException
	 * @throws CSSException
	 */
	public void addStyleSheet(InputSource cssSrc) 
	throws DOMException, IOException, CSSException {
		getStyleSheet().parseCSSStyleSheet(cssSrc);
	}

	/*
	 * This method should only be called from HeadElement.
	 */
	void onEmbeddedStyleAdd(StyleDefiner element) {
		if(element instanceof LinkElement) {
			linkedStyle.add((LinkElement)element);
		} else if(element instanceof StyleElement) {
			embeddedStyle.add((StyleElement)element);
		}
		onStyleModify();
	}

	/*
	 * This method should only be called from HeadElement.
	 */
	void onEmbeddedStyleRemove(StyleDefiner element) {
		if(element instanceof LinkElement){
			linkedStyle.remove(element);
		} else if(element instanceof StyleElement){
			embeddedStyle.remove(element);
		}
		onStyleModify();
	}

	/**
	 * Notifies the document about any change in style.
	 * 
	 */
	void onStyleModify() {
		if(mergedStyleSheet != null){
			mergedStyleSheet = null;
			styleCacheSerial++;
		}
	}

	/**
	 * Gets the serial number for the document-wide merged style sheet.
	 * <p>
	 * The serial number will be increased by one each time that any of 
	 * the sheets that conform the style is changed.
	 * 
	 * @return the serial number for the merged style sheet.
	 */
	int getStyleCacheSerial() {
		return styleCacheSerial;
	}

	/**
	 * Gets the default CSS style sheet to be used by this document.
	 * 
	 * @return the default style sheet.
	 */
	public DOM4JCSSStyleSheet getDefaultStyleSheet() {
		return defStyleSheet;
	}

	/**
	 * Sets the default CSS style sheet to be used by this document.
	 * 
	 * @param styleSheet the default style sheet.
	 */
	public void setDefaultStyleSheet(DOM4JCSSStyleSheet styleSheet) {
		this.defStyleSheet = styleSheet;
		onStyleModify();
	}

	/**
	 * Sets the CSS style declaration defined by the end-user as of "important" priority.
	 * 
	 * @param style the user "important" style declaration.
	 */
	public void setUserImportantStyleDeclaration(BaseCSSStyleDeclaration style) {
		this.userImportantStyle = style;
		onStyleModify();
	}

	/**
	 * Gets the style database that applies to the styles in this 
	 * document.
	 * 
	 * @return the style database.
	 */
	public StyleDatabase getStyleDatabase() {
		return styleDb;
	}

	/**
	 * Set the style database to be applied to all styles.
	 * 
	 * @param styleDb the style database.
	 */
	public void setStyleDatabase(StyleDatabase styleDb) {
		this.styleDb = styleDb;
		onStyleModify();
	}

	/**
	 * Gets the Base URL of this Document.
	 * 
	 * @return the base URL.
	 */
	public URL getBaseURL() {
		return baseURL;
	}

	/**
	 * Sets the Base URL of this Document.
	 * 
	 * @param baseURL the base URL.
	 */
	public void setBaseURL(URL baseURL) {
		this.baseURL = baseURL;
	}

	/**
	 * Gets an URL for the given URI, taking into account the 
	 * Base URL if appropriate.
	 * 
	 * @param uri the uri.
	 * @return the absolute URL.
	 * @throws MalformedURLException if the uri was wrong.
	 */
	public URL getURL(String uri) throws MalformedURLException {
		if(uri.length() == 0){
			throw new MalformedURLException("Empty URI");
		}
		URL url;
		if(uri.indexOf("//") < 0){
			url = new URL(getBaseURL(), uri);
		} else {
			url = new URL(uri);
		}
		return url;
	}

	/**
	 * Opens a connection for the given URI, taking into account the 
	 * Base URL if needed.
	 * 
	 * @param uri the uri to open a connection.
	 * @return the URL connection.
	 * @throws IOException if the uri was wrong, or the stream could 
	 * not be opened.
	 */
	public URLConnection openConnection(String uri) throws IOException {
		return getURL(uri).openConnection();
	}

	/**
	 * Opens an InputStream for the given URI, taking into account the 
	 * Base URL if needed.
	 * 
	 * @param uri the uri to open a connection.
	 * @return the InputStream.
	 * @throws IOException if the uri was wrong, or the stream could 
	 * not be opened.
	 */
	public InputStream openStream(String uri) throws IOException {
		return getURL(uri).openStream();
	}

}
