/*

 Copyright (c) 2005-2011, Carlos Amengual.

 Licensed under a BSD-style License. You can find the license here:
 http://www.informatica.info/projects/css/LICENSE.txt

 */

package info.informatica.doc.pdf.itext;

import info.informatica.doc.style.css.AbstractStyleDatabase;

import org.w3c.dom.DOMException;
import org.w3c.dom.css.CSSPrimitiveValue;

import com.itextpdf.text.Font;
import com.itextpdf.text.PageSize;
import com.itextpdf.text.Rectangle;
import com.itextpdf.text.Font.FontFamily;

/**
 * PDF/iText Style database.
 * 
 * @author Carlos Amengual (amengual at informatica.info)
 * 
 */
public class PDFStyleDatabase extends AbstractStyleDatabase {

	protected Rectangle pageSize = null;

	/**
	 * Creates a style database for a PDF of A4 size.
	 *
	 */
	public PDFStyleDatabase() {
		this(PageSize.A4);
	}

	public PDFStyleDatabase(Rectangle pageSize) {
		super();
		this.pageSize = pageSize;
	}

	/**
	 * Sets the page size for this PDF.
	 * 
	 * @param pageSize the page size.
	 */
	public void setPageSize(Rectangle pageSize) {
		this.pageSize = pageSize;
	}

	public String getDefaultGenericFontFamily(String genericFamily) {
		String fontName = null;
		if (genericFamily.equals("serif")) {
			fontName = "Times-Roman";
		} else if (genericFamily.equals("sans serif")) {
			fontName = "Helvetica";
		} else if (genericFamily.equals("monospace")) {
			fontName = "Courier";
		} else if (genericFamily.equals("fantasy")) {
			fontName = "ZapfDingbats";
		}
		return fontName;
	}

	public boolean isFontFamilyAvailable(String fontFamily) {
		return !Font.getFamily(fontFamily).equals(FontFamily.UNDEFINED);
	}

	public int getFontSizeFromIdentifier(String familyName,
			String fontSizeIdentifier) throws DOMException {
		int sz;
		if (fontSizeIdentifier.equals("xx-small")) {
			sz = 8;
		} else if (fontSizeIdentifier.equals("x-small")) {
			sz = 9;
		} else if (fontSizeIdentifier.equals("small")) {
			sz = 10;
		} else if (fontSizeIdentifier.equals("medium")) {
			sz = 12;
		} else if (fontSizeIdentifier.equals("large")) {
			sz = 14;
		} else if (fontSizeIdentifier.equals("x-large")) {
			sz = 16;
		} else if (fontSizeIdentifier.equals("xx-large")) {
			sz = 18;
		} else {
			throw new DOMException(DOMException.INVALID_ACCESS_ERR,
					"Unknown size identifier: " + fontSizeIdentifier);
		}
		return sz;
	}

	public float getDocumentHeight() {
		return pageSize.getHeight();
	}

	public short getFontSizeUnit() {
		return CSSPrimitiveValue.CSS_PT;
	}

	public short getNaturalUnit() {
		return CSSPrimitiveValue.CSS_PT;
	}

	public float getDocumentWidth() {
		return pageSize.getWidth();
	}

	public float getWidthSize(String widthIdentifier)
			throws DOMException {
		if ("thin".equals(widthIdentifier)) {
			return 0.1f;
		} else if ("thick".equals(widthIdentifier)) {
			return 1.2f;
		} else if ("medium".equals(widthIdentifier)) {
			return 0.8f;
		} else {
			throw new DOMException(DOMException.SYNTAX_ERR,
					"Unknown identifier " + widthIdentifier);
		}
	}

	protected float cmToPixels(float cm) {
		/*
		 * PDFStyleDatabase should not need these methods anyway.
		 */
		// XXX
		double width = 1024;
		double height = 768;
		double diag = Math.sqrt(width * width + height * height);
		// Assume 17" = 43cm
		return (float) (cm * diag / 43.18);
	}

	protected float pxTocm(int px) {
		/*
		 * PDFStyleDatabase should not need these methods anyway.
		 */
		// XXX
		double width = 1024;
		double height = 768;
		double diag = Math.sqrt(width * width + height * height);
		// Assume 17" = 43cm
		return (float) (px * 43.18 / diag);
	}

}
