/*

 Copyright (c) 2005-2007, Carlos Amengual.

 Licensed under a BSD-style License. You can find the license here:
 http://www.informatica.info/projects/css/LICENSE.txt

 */

package info.informatica.doc.style.css;

import info.informatica.doc.style.css.property.CSSNumberValue;

import java.net.URL;

import org.w3c.dom.Node;
import org.w3c.dom.css.CSSPrimitiveValue;
import org.w3c.dom.css.CSSValue;

/**
 * Interface that allows the retrieval of the primitive, "computed" values 
 * of CSS properties for practical use in formatting.
 * <p>
 * Classes implementing this interface must inherit the elements as necessary to
 * get the final primitive values and not relative 'inherited' ones. Whenever possible, 
 * values intended to be 'used' (as opposed to simply 'computed' values) should be 
 * returned. See paragraph 6.1 of the CSS specification for the differences between 
 * 'computed' and 'used' values.
 * </p>
 * 
 * @author Carlos Amengual (amengual at informatica.info)
 * 
 */
public interface CSS2ComputedProperties {

    /**
     * Retrieves the computed object representation of the value of a CSS 
     * property. 
     * This method returns <code>null</code> if the property is a shorthand 
     * property. 
     * @param propertyName The name of the CSS property.
     * @return the object value of the property. 
     */
	public CSSValue getPropertyCSSValue(String propertyName);

    /**
     * Retrieves the String value of a CSS property.
     *  
     * @param propertyName The name of the CSS property. See the CSS property 
     *   index. 
     * @return  the value of the property. 
     */
	public String getPropertyValue(String propertyName);

	/**
	 * Computes the value, in the device's natural unit, of the given 
	 * number value.
	 * <p>
	 * Usage note: after calling this method, check whether the CSS 
	 * value is a percentage, and if the returned float is different 
	 * from zero, then compute the percentage base and apply to it.
	 * </p>
	 * @param cssValue the CSS value representing a number.
	 * @return the float value in the device's natural unit, or the 
	 * value of a percentage if the value is a percentage.
	 */
	public float computeFloatValue(CSSNumberValue cssValue);

	/**
	 * Gets the primitive, computed value for the 'color' property.
	 * 
	 * @return the value for the 'color' property.
	 */
	public CSSPrimitiveValue getColor();

	/**
	 * Gets the primitive, computed value for the 'background-color' property.
	 * 
	 * @return the value for the 'background-color' property.
	 */
	public CSSPrimitiveValue getBackgroundColor();

	/**
	 * Gets the computed value for the 'background-image' property.
	 * 
	 * @return the value for the 'background-image' property, or null if 
	 * no background image was set for the element.
	 */
	public String getBackgroundImage();

	/**
	 * Gets the 'used' value for the font-family property.
	 * <p>This method requires a style database.</p>
	 * 
	 * @return the value of the font-family property.
	 * @throws IllegalStateException if the style database has not been set.
	 */
	public String getFontFamily();

	/**
	 * Gets the computed font weight.
	 * 
	 * @return the font weight.
	 */
	public String getFontWeight();

	/**
	 * Gets the computed value of the font-size property.
	 * <p>
	 * May require a style database to work.
	 * </p>
	 * 
	 * @return the value of the font-size property, in typographic points.
	 */
	public int getFontSize();

	/**
	 * Gets the base URL that should be used to resolve relative URLs 
	 * in property values.
	 * 
	 * @return the base URL, or null if could not be 
	 * determined.
	 */
	public URL getBaseURL();

	/**
	 * Gets the peer node.
	 * 
	 * @return the peer node.
	 */
	public Node getPeerNode();

	/**
	 * Gets the XPath of the node to which this style applies.
	 * 
	 * @return the XPath of the node.
	 */
	public String getPeerXPath();

	/**
	 * Gets the computed style for the parent element.
	 * 
	 * @return the computed style for the parent element, or null if 
	 * there is no parent element, or has no style associated.
	 */
	public CSS2ComputedProperties getParentComputedStyle();

	/**
	 * Gets the style database used to compute the style.
	 * 
	 * @return the style database.
	 */
	public StyleDatabase getStyleDatabase();

}
