/*

 Copyright (c) 2005-2007, Carlos Amengual.

 Licensed under a BSD-style License. You can find the license here:
 http://www.informatica.info/projects/css/LICENSE.txt

 */

package info.informatica.doc.style.css;

import org.w3c.dom.DOMException;
import org.w3c.dom.css.CSSPrimitiveValue;

/**
 * CSS Style database.
 * <p>
 * To use CSS in practice, you need to have information about the target 
 * rendering device: available fonts, page sizes, etc. This information is 
 * supplied by an instance of the <code>StyleDatabase</code> interface.
 * </p>
 * <p>The appropriate implementation must be supplied to the style declaration, 
 * through the <code>StyleDatabaseAware</code> interface.
 * </p>
 * <p>This interface is a work in progress.</p>
 * 
 * @author Carlos Amengual (amengual at informatica.info)
 * @see StyleDatabaseAware
 */
public interface StyleDatabase {

	/**
	 * Gives the initial (default) value for the 'color' property 
	 * on this device.
	 * 
	 * @return the default color value.
	 */
	public CSSPrimitiveValue getInitialColor();

	/**
	 * Sets the initial (default) value for the 'color' property 
	 * on this device.
	 * 
	 * @param initialColor the String representing default color value.
	 */
	public void setInitialColor(String initialColor);

	/**
	 * Gets the name of the default font used when a generic font family (serif,
	 * sans-serif, monospace, cursive, fantasy) is specified.
	 * 
	 * @param genericFamily
	 *            the name of the logical font.
	 * @return the name of the real font to which the generic name is mapped to,
	 *         or null if none.
	 */
	public String getDefaultGenericFontFamily(String genericFamily);

	/**
	 * Gets the real name of the default font.
	 * <p>
	 * For example, if the default generic font name is 'serif', this 
	 * method should return the same as getDefaultGenericFontFamily("serif").
	 * 
	 * @return the name of the default real font.
	 */
	public String getDefaultGenericFontFamily();

	/**
	 * Checks if a font family is available.
	 * 
	 * @param fontFamily
	 *            the font family name.
	 * @return true if the font is available, false otherwise.
	 */
	public boolean isFontFamilyAvailable(String fontFamily);

	/**
	 * Gets the font size from the given size identifier (small, 
	 * medium, etc.), expressed in typographic points.
	 * 
	 * @param familyName the font family name.
	 * @param fontSizeIdentifier the font size identifier.
	 * @return the font size.
	 * @throws DOMException
	 */
	public int getFontSizeFromIdentifier(String familyName,
			String fontSizeIdentifier) throws DOMException;

	/**
	 * Gets the identifier of the device's natural unit.
	 *  
	 * @return the unit identifier as in <code>CSSPrimitiveValue</code>.
	 */
	public short getNaturalUnit();

	/**
	 * Makes an unit conversion (for the units known to this device).
	 * 
	 * @param initialValue the value to be converted, expressed in the initial unit.
	 * @param initialUnitType the initial unit type.
	 * @param destUnitType the destination unit type.
	 * @return the value, expressed in units of destUnitType.
	 * @throws DOMException
	 */
	public float floatValueConversion(float initialValue,
			short initialUnitType, short destUnitType) throws DOMException;

	/**
	 * Makes an unit conversion to the natural unit of this device.
	 * 
	 * @param initialValue the value to be converted, expressed in the initial unit.
	 * @param initialUnitType the initial unit type.
	 * @return the value, expressed in units of destUnitType.
	 * @throws DOMException
	 */
	public float floatValueConversion(float initialValue,
			short initialUnitType) throws DOMException;

	/**
	 * Gives the size of the 'ex' unit, expressed in 'pt' 
	 * (typographic points) unit.
	 * 
	 * @param familyName the font family name.
	 * @param size the font size.
	 * @return the size of the 'ex' unit, in units of 'pt'.
	 */
	public int getExSizeInPt(String familyName, int size);

	/**
	 * Gets the size corresponging to the given identifier (thin, thick, 
	 * medium).
	 * 
	 * @param widthIdentifier the CSS width identifier.
	 * @return the size.
	 * @throws DOMException
	 */
	public float getWidthSize(String widthIdentifier) throws DOMException;

	/**
	 * Gets the height of the document (or screen viewport) in the natural 
	 * unit for the device.
	 * 
	 * @return the height of the document.
	 */
	public float getDocumentHeight();

	/**
	 * Gets the width of the document (or screen viewport) in the natural 
	 * unit for the device.
	 * 
	 * @return the width of the document.
	 */
	public float getDocumentWidth();

	/**
	 * Gets the width of the scrollbar that appears when a box 
	 * overflows with scroll.
	 * 
	 * @return the width of the scrollbar.
	 */
	public float getScrollbarWidth();

}