/*

 Copyright (c) 2005-2011, Carlos Amengual.

 Licensed under a BSD-style License. You can find the license here:
 http://www.informatica.info/projects/css/LICENSE.txt

 */

package info.informatica.doc.style.css.dom;

import info.informatica.doc.style.css.SACParserFactory;

import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.util.Iterator;

import org.w3c.css.sac.CSSException;
import org.w3c.css.sac.DocumentHandler;
import org.w3c.css.sac.InputSource;
import org.w3c.css.sac.LexicalUnit;
import org.w3c.css.sac.Parser;
import org.w3c.css.sac.SACMediaList;
import org.w3c.css.sac.SelectorList;
import org.w3c.dom.DOMException;
import org.w3c.dom.css.CSSMediaRule;
import org.w3c.dom.css.CSSRule;
import org.w3c.dom.css.CSSRuleList;
import org.w3c.dom.css.CSSStyleSheet;
import org.w3c.dom.stylesheets.MediaList;

/**
 * Implementation of CSSMediaRule.
 * 
 * @author Carlos Amengual (amengual at informatica.info)
 * 
 */
public class DOMCSSMediaRule extends BaseCSSRule implements CSSMediaRule {

	private MediaList mediaList = null;

	private CSSRuleArrayList cssRules;

	public DOMCSSMediaRule(CSSStyleSheet parentSheet) {
		super(parentSheet, CSSRule.MEDIA_RULE);
		mediaList = new DOMMediaList();
		cssRules = new CSSRuleArrayList();
	}

	public MediaList getMedia() {
		return mediaList;
	}

	public CSSRuleList getCssRules() {
		return cssRules;
	}

	public int insertRule(String rule, int index) throws DOMException {
		if (index < 0 || index > cssRules.size()) {
			throw new DOMException(DOMException.INDEX_SIZE_ERR,
					"Index out of bounds in media list");
		}
		Parser parser = SACParserFactory.createSACParser();
		InputSource source = new InputSource();
		Reader re = new StringReader(rule);
		source.setCharacterStream(re);
		RulesetDocumentHandler handler = new RulesetDocumentHandler();
		handler.setCurrentInsertionIndex(index);
		parser.setDocumentHandler(handler);
		try {
			parser.parseStyleDeclaration(source);
		} catch (CSSException e) {
			throw new DOMException(DOMException.INVALID_ACCESS_ERR, e
					.getMessage());
		} catch (IOException e) {
			// This should never happen!
			throw new DOMException(DOMException.INVALID_STATE_ERR, e
					.getMessage());
		}
		return index;
	}

	public void deleteRule(int index) throws DOMException {
		if (index < 0 || index >= cssRules.size()) {
			throw new DOMException(DOMException.INDEX_SIZE_ERR,
					"Could not delete rule in media list: index out of bounds.");
		}
		cssRules.remove(index);
	}

	public int insertRule(CSSRule cssrule, int index) {
		return cssRules.insertRule(cssrule, index);
	}
	
	public int addRule(CSSRule cssrule) {
		return insertRule(cssrule, cssRules.getLength());
	}

	@Override
	public String getCssText() {
		StringBuilder sb = new StringBuilder(30 + cssRules.getLength() * 20);
		sb.append("@media ").append(mediaList.getMediaText()).append(" {");
		Iterator<CSSRule> it = cssRules.iterator();
		while (it.hasNext()) {
			sb.append(it.next().getCssText());
		}
		sb.append('}');
		return sb.toString();
	}

	@Override
	public void setCssText(String cssText) throws DOMException {
		throw new UnsupportedOperationException();
	}

	public class RulesetDocumentHandler implements DocumentHandler {
		private CSSRule currentRule = null;

		private int currentInsertionIndex = 0;
		
		private boolean active = false;

		public RulesetDocumentHandler() {
			super();
		}

		public void setCurrentInsertionIndex(int index) {
			currentInsertionIndex = index;
		}

		public void startDocument(InputSource source) throws CSSException {
			currentRule = null;
			currentInsertionIndex = 0;
			active = true;
		}

		public void endDocument(InputSource source) throws CSSException {
		}

		public void comment(String text) throws CSSException {
		}

		public void ignorableAtRule(String atRule) throws CSSException {
		}

		public void namespaceDeclaration(String prefix, String uri)
				throws CSSException {
		}

		public void importStyle(String uri, SACMediaList media,
				String defaultNamespaceURI) throws CSSException {
			// Ignore any '@import' rule that occurs inside a block (CSS 2.1 §4.1.5)
		}

		public void startMedia(SACMediaList media) throws CSSException {
			// Nested @media rule: ignore
			active = false;
		}

		public void endMedia(SACMediaList media) throws CSSException {
			active = true;
		}

		public void startPage(String name, String pseudo_page)
				throws CSSException {
			if(active) {
				currentRule = new DOMCSSPageRule(getParentStyleSheet());
			}
		}

		public void endPage(String name, String pseudo_page)
				throws CSSException {
			if(active) {
				currentInsertionIndex = insertRule(currentRule,
						currentInsertionIndex);
				currentRule = null;
			}
		}

		public void startFontFace() throws CSSException {
			if(active) {
				currentRule = new DOMCSSFontFaceRule(getParentStyleSheet());
			}
		}

		public void endFontFace() throws CSSException {
			if(active) {
				currentInsertionIndex = insertRule(currentRule,
						currentInsertionIndex);
				currentRule = null;
			}
		}

		public void startSelector(SelectorList selectors) throws CSSException {
			if(active) {
				if (currentRule == null) {
					currentRule = new DOMCSSStyleRule(getParentStyleSheet());
				}
				((CSSStyleDeclarationRule) currentRule).setSelectorList(selectors);
			}
		}

		public void endSelector(SelectorList selectors) throws CSSException {
			if(active) {
				if (currentRule instanceof DOMCSSStyleRule) {
					insertRule(currentRule, currentInsertionIndex);
					currentRule = null;
				}
			}
		}

		public void property(String name, LexicalUnit value, boolean important)
				throws CSSException {
			if(active) {
				String importantString = null;
				if (important) {
					importantString = "important";
				}
				((BaseCSSStyleDeclaration) ((CSSStyleDeclarationRule) currentRule)
						.getStyle()).setProperty(name, value, importantString);
			}
		}
	}
}
