/*

 Copyright (c) 2005-2011, Carlos Amengual.

 Licensed under a BSD-style License. You can find the license here:
 http://www.informatica.info/projects/css/LICENSE.txt

 */

package info.informatica.doc.style.css.dom;

import java.util.Comparator;

import org.w3c.css.sac.Condition;
import org.w3c.css.sac.ConditionalSelector;
import org.w3c.css.sac.DescendantSelector;
import org.w3c.css.sac.Selector;
import org.w3c.css.sac.SiblingSelector;
import org.w3c.dom.css.CSSRule;
import org.w3c.dom.css.CSSStyleRule;
import org.w3c.dom.css.CSSStyleSheet;

/**
 * CSS style rule.
 * 
 * @author Carlos Amengual (amengual at informatica.info)
 * 
 */
class DOMCSSStyleRule extends CSSStyleDeclarationRule implements
		CSSStyleRule {

	public DOMCSSStyleRule(CSSStyleSheet parentSheet) {
		super(parentSheet, CSSRule.STYLE_RULE);
	}

	public DOMCSSStyleRule() {
		this(null);
	}

	/**
	 * Returns the specificity of this rule.
	 * <p>
	 * See Cascading Style Sheets, level 2 revision 1 CSS 2.1 Specification,
	 * paragraph 6.4.3.
	 * 
	 * @return the specificity.
	 */
	Specifity getSpecifity(int index) {
		return new Specifity(getSelectorList().item(index));
	}

	/**
	 * The specificity of this rule.
	 * <p>
	 * See Cascading Style Sheets, level 2 revision 1 CSS 2.1 Specification,
	 * paragraph 6.4.3.
	 */
	class Specifity {
		short id_count = 0;

		short attrib_classes_count = 0;

		short names_pseudoelements_count = 0;

		public Specifity(Selector selector) {
			super();
			specifity(selector);
		}

		private void specifity(Selector selector) {
			switch (selector.getSelectorType()) {
			case Selector.SAC_ELEMENT_NODE_SELECTOR:
			case Selector.SAC_PSEUDO_ELEMENT_SELECTOR:
				names_pseudoelements_count++;
				break;
			case Selector.SAC_CONDITIONAL_SELECTOR:
				Condition cond = ((ConditionalSelector) selector)
						.getCondition();
				switch (cond.getConditionType()) {
				case Condition.SAC_CLASS_CONDITION:
				case Condition.SAC_ATTRIBUTE_CONDITION:
				case Condition.SAC_ONE_OF_ATTRIBUTE_CONDITION:
				case Condition.SAC_PSEUDO_CLASS_CONDITION:
					attrib_classes_count++;
					break;
				case Condition.SAC_ID_CONDITION:
					id_count++;
					break;
				}
				specifity(((ConditionalSelector) selector).getSimpleSelector());
				break;
			case Selector.SAC_DESCENDANT_SELECTOR:
			case Selector.SAC_CHILD_SELECTOR:
				specifity(((DescendantSelector) selector).getSimpleSelector());
				specifity(((DescendantSelector) selector).getAncestorSelector());
				break;
			case Selector.SAC_DIRECT_ADJACENT_SELECTOR:
				specifity(((SiblingSelector) selector).getSiblingSelector());
				specifity(((SiblingSelector) selector).getSelector());
				break;
			}
		}

		/**
		 * Gets the style rule to which this specificity aplies.
		 * 
		 * @return the style rule.
		 */
		public DOMCSSStyleRule getCSSStyleRule() {
			return DOMCSSStyleRule.this;
		}
	}

	static class SpecificityComparator implements Comparator<Specifity> {

		public int compare(Specifity o1, Specifity o2) {
			return selectorCompare(o1, o2) * 4096
					+ o1.getCSSStyleRule().insertionIndex
					- o2.getCSSStyleRule().insertionIndex;
		}

		public int selectorCompare(Specifity o1, Specifity o2) {
			return (o1.id_count - o2.id_count)
					* 2048
					+ (o1.attrib_classes_count - o2.attrib_classes_count)
					* 128
					+ (o1.names_pseudoelements_count - o2.names_pseudoelements_count);
		}
	}
}
